#define CONFIG_YAFFS_ECC_WRONG_ORDER 1

/*
 * YAFFS: Yet another FFS. A NAND-flash specific file system. 
 *
 * yaffs_ecc.c: ECC generation/correction algorithms.
 *
 * Copyright (C) 2002 Aleph One Ltd.
 *
 * Created by Charles Manning <charles@aleph1.co.uk>
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 */
 
 /*
 * This code implements the ECC algorithm used in SmartMedia.
 *
 * The ECC comprises 22 bits of parity information and is stuffed into 3 bytes. 
 * The two unused bit are set to 1.
 * The ECC can correct single bit errors in a 256-byte page of data. Thus, two such ECC 
 * blocks are used on a 512-byte NAND page.
 *
 */

// Table generated by gen-ecc.c
// Using a table means we do not have to calculate p1..p4 and p1'..p4'
// for each byte of data. These are instead provided in a table in bits7..2.
// Bit 0 of each entry indicates whether the entry has an odd or even parity, and therefore
// this bytes influence on the line parity.

#include "main.h"
#include "nand.h"
#include "yaffs.h"

#include "util.h"
#include "serial.h"

static const unsigned char column_parity_table[] = {
0x00, 0x55, 0x59, 0x0c, 0x65, 0x30, 0x3c, 0x69, 0x69, 0x3c, 0x30, 0x65, 0x0c, 0x59, 0x55, 0x00, 
0x95, 0xc0, 0xcc, 0x99, 0xf0, 0xa5, 0xa9, 0xfc, 0xfc, 0xa9, 0xa5, 0xf0, 0x99, 0xcc, 0xc0, 0x95, 
0x99, 0xcc, 0xc0, 0x95, 0xfc, 0xa9, 0xa5, 0xf0, 0xf0, 0xa5, 0xa9, 0xfc, 0x95, 0xc0, 0xcc, 0x99, 
0x0c, 0x59, 0x55, 0x00, 0x69, 0x3c, 0x30, 0x65, 0x65, 0x30, 0x3c, 0x69, 0x00, 0x55, 0x59, 0x0c, 
0xa5, 0xf0, 0xfc, 0xa9, 0xc0, 0x95, 0x99, 0xcc, 0xcc, 0x99, 0x95, 0xc0, 0xa9, 0xfc, 0xf0, 0xa5, 
0x30, 0x65, 0x69, 0x3c, 0x55, 0x00, 0x0c, 0x59, 0x59, 0x0c, 0x00, 0x55, 0x3c, 0x69, 0x65, 0x30, 
0x3c, 0x69, 0x65, 0x30, 0x59, 0x0c, 0x00, 0x55, 0x55, 0x00, 0x0c, 0x59, 0x30, 0x65, 0x69, 0x3c, 
0xa9, 0xfc, 0xf0, 0xa5, 0xcc, 0x99, 0x95, 0xc0, 0xc0, 0x95, 0x99, 0xcc, 0xa5, 0xf0, 0xfc, 0xa9, 
0xa9, 0xfc, 0xf0, 0xa5, 0xcc, 0x99, 0x95, 0xc0, 0xc0, 0x95, 0x99, 0xcc, 0xa5, 0xf0, 0xfc, 0xa9, 
0x3c, 0x69, 0x65, 0x30, 0x59, 0x0c, 0x00, 0x55, 0x55, 0x00, 0x0c, 0x59, 0x30, 0x65, 0x69, 0x3c, 
0x30, 0x65, 0x69, 0x3c, 0x55, 0x00, 0x0c, 0x59, 0x59, 0x0c, 0x00, 0x55, 0x3c, 0x69, 0x65, 0x30, 
0xa5, 0xf0, 0xfc, 0xa9, 0xc0, 0x95, 0x99, 0xcc, 0xcc, 0x99, 0x95, 0xc0, 0xa9, 0xfc, 0xf0, 0xa5, 
0x0c, 0x59, 0x55, 0x00, 0x69, 0x3c, 0x30, 0x65, 0x65, 0x30, 0x3c, 0x69, 0x00, 0x55, 0x59, 0x0c, 
0x99, 0xcc, 0xc0, 0x95, 0xfc, 0xa9, 0xa5, 0xf0, 0xf0, 0xa5, 0xa9, 0xfc, 0x95, 0xc0, 0xcc, 0x99, 
0x95, 0xc0, 0xcc, 0x99, 0xf0, 0xa5, 0xa9, 0xfc, 0xfc, 0xa9, 0xa5, 0xf0, 0x99, 0xcc, 0xc0, 0x95, 
0x00, 0x55, 0x59, 0x0c, 0x65, 0x30, 0x3c, 0x69, 0x69, 0x3c, 0x30, 0x65, 0x0c, 0x59, 0x55, 0x00, 
};

void yaffs_ECCCalculate(const unsigned char *data,unsigned char *ecc)
{
	unsigned int i;
	
	unsigned char col_parity = 0;
	unsigned char line_parity = 0;
	unsigned char line_parity_prime = 0;
	unsigned char t;
	unsigned char b;
	
	for(i = 0; i < 256; i++)
	{
		b = column_parity_table[*data++];
		col_parity ^= b;

		if(b & 0x01) // odd number of bits in the byte
		{
			line_parity ^= i;
			line_parity_prime ^= ~i;
		}
		
	}
	
	ecc[2] = (~col_parity) | 0x03;
	
	t = 0;
	if(line_parity       & 0x80) t |= 0x80;
	if(line_parity_prime & 0x80) t |= 0x40;
	if(line_parity       & 0x40) t |= 0x20;
	if(line_parity_prime & 0x40) t |= 0x10;
	if(line_parity       & 0x20) t |= 0x08;
	if(line_parity_prime & 0x20) t |= 0x04;
	if(line_parity       & 0x10) t |= 0x02;
	if(line_parity_prime & 0x10) t |= 0x01;
	ecc[1] = ~t;
	
	t = 0;
	if(line_parity       & 0x08) t |= 0x80;
	if(line_parity_prime & 0x08) t |= 0x40;
	if(line_parity       & 0x04) t |= 0x20;
	if(line_parity_prime & 0x04) t |= 0x10;
	if(line_parity       & 0x02) t |= 0x08;
	if(line_parity_prime & 0x02) t |= 0x04;
	if(line_parity       & 0x01) t |= 0x02;
	if(line_parity_prime & 0x01) t |= 0x01;
	ecc[0] = ~t;

#if CONFIG_YAFFS_ECC_WRONG_ORDER
	// Swap the bytes into the wrong order
	t = ecc[0];
	ecc[0] = ecc[1];
	ecc[1] = t;
#endif 
}

int yaffs_ECCCorrect(unsigned char *data, unsigned char *read_ecc, const unsigned char *test_ecc)
{
	unsigned char d0, d1, d2; // deltas 

	d0 = read_ecc[0] ^ test_ecc[0];
	d1 = read_ecc[1] ^ test_ecc[1];
	d2 = read_ecc[2] ^ test_ecc[2];
	
	
	
	if((d0 | d1 | d2) == 0)
	{
		// no error
		return 0;
	}
	
	if( ((d0 ^ (d0 >> 1)) & 0x55) == 0x55 &&
	    ((d1 ^ (d1 >> 1)) & 0x55) == 0x55 &&
	    ((d2 ^ (d2 >> 1)) & 0x54) == 0x54)
	{
		// Single bit (recoverable) error in data

		unsigned byte;
		unsigned bit;

#if CONFIG_YAFFS_ECC_WRONG_ORDER
		// swap the bytes to correct for the wrong order
		unsigned char t;
		
		t = d0;
		d0 = d1;
		d1 = t;
#endif
		
		bit = byte = 0;
		
		
		if(d1 & 0x80) byte |= 0x80;
		if(d1 & 0x20) byte |= 0x40;
		if(d1 & 0x08) byte |= 0x20;
		if(d1 & 0x02) byte |= 0x10;
		if(d0 & 0x80) byte |= 0x08;
		if(d0 & 0x20) byte |= 0x04;
		if(d0 & 0x08) byte |= 0x02;
		if(d0 & 0x02) byte |= 0x01;

		if(d2 & 0x80) bit |= 0x04;
		if(d2 & 0x20) bit |= 0x02;
		if(d2 & 0x08) bit |= 0x01;

		data[byte] ^= (1 << bit);
		
		return 1;
	}
	
	if((yaffs_CountBits(d0)+yaffs_CountBits(d1)+yaffs_CountBits(d2)) == 1)
	{
		// Reccoverable error in ecc
		
		read_ecc[0] = test_ecc[0];
		read_ecc[1] = test_ecc[1];
		read_ecc[2] = test_ecc[2];
		
		return 1;
	}
	
	// Unrecoverable error
	
	return -1;
	    

}

static const char yaffs_countBitsTable[256] =
{
	0,1,1,2,1,2,2,3,1,2,2,3,2,3,3,4,
	1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
	1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
	2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
	1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
	2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
	2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
	3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
	1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
	2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
	2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
	3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
	2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
	3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
	3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
	4,5,5,6,5,6,6,7,5,6,6,7,6,7,7,8
};

int yaffs_CountBits(int x)
{
	int retVal;
	retVal = yaffs_countBitsTable[x];
	return retVal;
}

int load_yaffs(int verbose, int startpage, int endpage) {
	int i, l;
	unsigned char buffer[528];
	int kerID=0, rdID=0;
	int kerSZ, rdSZ;
	unsigned char *sers, eccfixed;
	int nandsize;

	sers = (unsigned char *)HEAP_RAM_BASE;
	for(i=0;i<0x10000;i++)sers[i]=120;
	eccfixed=0;
		
	/* Find type of NAND */
	nandsize=0;
	i=nand_read_id();
	i&=0xffff;
	if(i==0xec75)nandsize=65536; else if(i==0xec76)nandsize=131072;
	SerialOutputHexNib(i, 4);
	SerialOutputString("\r\n");
	
	/* Check for invalid blocks */
	l=0;
	for(i=0;i<nandsize;i+=32) {
		int k;
		k=0;
		nand_read_oob(i, buffer+512, nandsize);
		if(buffer[517]!=0xff) {
			l++; 
			k=1;
		} else {
			nand_read_oob(i+1, buffer+512, nandsize);
			if(buffer[517]!=0xff) {
				l++;
				k=1;
			}
		}
		if(k) {
			SerialOutputString("Block ");
			SerialOutputHexNib(i>>5, 4);
			SerialOutputString(" invalid.\r\n");
		}
	}
	
	SerialOutputString("Found ");
	SerialOutputDec(l);
	SerialOutputString(" invalid blocks.\r\n");
	
	/* Look for zImage and ramdisk */
	for(i=startpage; i<endpage; i++) {
		yaffs_TagsUnion tags;
		
		nand_read_oob(i, buffer+512, nandsize);

		/* Check for invalid block */
		if((i&31)==0 || (i&31)==1) {
			if(buffer[517]!=0xff) {
				i|=31;
				continue;
			}
		}

		tags.asBytes[0]=buffer[512];
		tags.asBytes[1]=buffer[513];
		tags.asBytes[2]=buffer[514];
		tags.asBytes[3]=buffer[515];
		tags.asBytes[4]=buffer[518];
		tags.asBytes[5]=buffer[519];
		tags.asBytes[6]=buffer[523];
		tags.asBytes[7]=buffer[524];

		if(tags.asTags.chunkId==0 && yaffs_CountBits(buffer[516])>4) {
			int l;
			SerialOutputHexNib(i, 5);
			SerialOutputString(":   ");
			SerialOutputHexNib(buffer[516],2);
			SerialOutputString(", chunkId:");

			nand_read(i, buffer, nandsize);
			l=buffer[292]+256*buffer[293]+65536*buffer[294]+16777216*buffer[295];
			SerialOutputHexNib(tags.asTags.chunkId,5);
			SerialOutputString("  serialNo:");
			SerialOutputHexNib(tags.asTags.serialNumber,1);
			SerialOutputString("  size:");
			SerialOutputHex(l);
			SerialOutputString("  objectID:");
			SerialOutputHexNib(tags.asTags.objectId,5);
			SerialOutputString("  name:");
			SerialOutputString(buffer+10);
			SerialOutputString("\r\n");
			if(!MyStrNCmp("zImage", buffer+10,6)) {
				kerID=tags.asTags.objectId;
				kerSZ=l;
			}
			if(!MyStrNCmp("sh3.gz", buffer+10,6)) {
				rdID=tags.asTags.objectId;
				rdSZ=l;
			}
		}
	}

	/* Load the files */
	for(i=startpage; i<endpage; i++) {
		yaffs_TagsUnion tags;
		unsigned char ecc[3], *ptr;
		
		nand_read_oob(i, buffer+512, nandsize);

		/* Check for invalid block */
		if((i&31)==0 || (i&31)==1) {
			if(buffer[517]!=0xff) {
				i|=31;
				continue;
			}
		}

		tags.asBytes[0]=buffer[512];
		tags.asBytes[1]=buffer[513];
		tags.asBytes[2]=buffer[514];
		tags.asBytes[3]=buffer[515];
		tags.asBytes[4]=buffer[518];
		tags.asBytes[5]=buffer[519];
		tags.asBytes[6]=buffer[523];
		tags.asBytes[7]=buffer[524];

		if(verbose) {
				SerialOutputHexNib(i, 5);
				SerialOutputString(":   ");
				
				SerialOutputString(",  DS:");
				
				SerialOutputHexNib(buffer[516],2);
				SerialOutputString(", chunkId:");
				SerialOutputHexNib(tags.asTags.chunkId,5);
			
				SerialOutputString("  serialNo:");
				SerialOutputHexNib(tags.asTags.serialNumber,1);
				SerialOutputString("  objectID:");
				SerialOutputHexNib(tags.asTags.objectId,5);
				SerialOutputString("\r\n");
		}

		if(tags.asTags.chunkId!=0 && yaffs_CountBits(buffer[516])>4) {
			int k=0;
			ptr=0;
			if(tags.asTags.objectId==kerID) {
				ptr = (unsigned char *)KERNEL_RAM_BASE;
				k=0;
			}
			if(tags.asTags.objectId==rdID) {
				ptr = (unsigned char *)RAMDISK_RAM_BASE;
				k=0x8000;
			}

			if(ptr && sers[k+tags.asTags.chunkId]!=120) {
				if( (tags.asTags.serialNumber==sers[k+tags.asTags.chunkId]-1) ||
						(tags.asTags.serialNumber==3 && sers[k+tags.asTags.chunkId]==0) )
					ptr=0;
				SerialOutputString("Redoing chunk ");
				SerialOutputHexNib(tags.asTags.chunkId,5);
				SerialOutputString("  objectID=");
				SerialOutputHexNib(tags.asTags.objectId,5);
				SerialOutputString("  serial=");
				SerialOutputHexNib(tags.asTags.serialNumber,1);
				SerialOutputString("  Old ser=");
				SerialOutputHexNib(sers[k+tags.asTags.chunkId],2);
				SerialOutputString("\r\n");
				
			}

			if(ptr) {
				ptr += 512 * (tags.asTags.chunkId-1);
				sers[k+tags.asTags.chunkId]=tags.asTags.serialNumber;
				nand_read(i, ptr, nandsize);
				yaffs_ECCCalculate(ptr, ecc);
				k=yaffs_ECCCorrect(ptr, buffer+520, ecc);
				if(k==-1) {
					SerialOutputString("Uncorrected error:  chunkId=");
					SerialOutputHexNib(tags.asTags.chunkId,5);
					SerialOutputString("  objectID=");
					SerialOutputHexNib(tags.asTags.objectId,5);
					SerialOutputString("\r\n");
				} else eccfixed+=k;
				yaffs_ECCCalculate(ptr+256, ecc);
				k=yaffs_ECCCorrect(ptr+256, buffer+525, ecc);
				if(k==-1) {
					SerialOutputString("Uncorrected error:  chunkId=");
					SerialOutputHexNib(tags.asTags.chunkId,5);
					SerialOutputString("  objectID=");
					SerialOutputHexNib(tags.asTags.objectId,5);
					SerialOutputString("\r\n");
				} else eccfixed+=k;

				if(verbose) {
					SerialOutputHexNib(i, 5);
					SerialOutputString(":   ");
				
					SerialOutputDec(k);
					SerialOutputString(",  DS:");
					
					SerialOutputHexNib(buffer[516],2);
					SerialOutputString(", chunkId:");
					SerialOutputHexNib(tags.asTags.chunkId,5);
				
					SerialOutputString("  serialNo:");
					SerialOutputHexNib(tags.asTags.serialNumber,1);
					SerialOutputString("  objectID:");
					SerialOutputHexNib(tags.asTags.objectId,5);
					SerialOutputString("\r\n");
				}
			}
		}
		
	}
	
	return 0;
}

