/*-------------------------------------------------------------------------
 * Filename:      serial.c
 * Version:       $Id: serial.c,v 1.18 2000/07/10 14:20:17 erikm Exp $
 * Copyright:     Copyright (C) 1999, Erik Mouw
 * Author:        Erik Mouw <J.A.K.Mouw@its.tudelft.nl>
 * Description:   Serial utilities for blob
 * Created at:    Tue Aug 24 20:25:00 1999
 * Modified by:   Erik Mouw <J.A.K.Mouw@its.tudelft.nl>
 * Modified at:   Mon Oct  4 20:11:14 1999
 *-----------------------------------------------------------------------*/
/*
 * serial.c: Serial utilities for blob
 *
 * Copyright (C) 1999  Erik Mouw (J.A.K.Mouw@its.tudelft.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "config.h"
#include "serial.h"
#include "time.h"
#include "util.h"
#include "crctab.h"

#define TDFE 0x20

/*
 * Output a single byte to the serial port.
 */
void SerialOutputByte(const char c)
{
    volatile unsigned char *SCFTDR2;
    volatile unsigned short *SCFSR2;
    
    SCFTDR2 = (unsigned char *)0xffe8000c;
    SCFSR2 = (unsigned short *)0xffe80010;

    while ( ((*SCFSR2) & TDFE) == 0 );
    
    *SCFTDR2 = c;
    *SCFSR2 = ~TDFE;
}

void SerialOutputString(const char *s) {
	while(*s != 0)
		SerialOutputByte(*s++);
}

void SerialOutputHexNib(const u32 h, int Nib)
{
	char c;
	int i;
	
	for(i = Nib - 1; i >= 0; i--) {
		c = (char)((h >> (i * 4)) & 0x0f);

		if(c > 9)
			c += ('A' - 10);
		else
			c += '0';

		SerialOutputByte(c);
	}
}

void SerialOutputHex(const u32 h) {
	SerialOutputHexNib(h, 8);
}


/*
 * Write the argument of the function in decimal to the serial port.
 * We just assume that each argument is positive (i.e. unsigned).
 */
void SerialOutputDec(const u32 d)
{
    	u32 dd=d;
        char b[11];

        if(dd==0) {
            SerialOutputByte('0');
        } else {
            int i;
            i=0;
            while(dd!=0) {
                b[i] = dd % 10;
                dd=dd/10;
                i++;
                if(i==10)dd=0;
            }
            while(i>0) {
                i--;
                SerialOutputByte('0'+b[i]);
            }
        }

}

#define RDF 3

/*
 * Read a single byte from the serial port. Returns 1 on success, 0
 * otherwise. When the function is succesfull, the character read is
 * written into its argument c.
 */
int SerialInputByte(char *c)
{
    unsigned char *SCFRDR2;
    volatile unsigned short *SCFSR2;
    
    SCFRDR2 = (unsigned char *)0xffe80014;
    SCFSR2 = (unsigned short *)0xffe80010;

    if ( (*SCFSR2) & RDF ) {
        *c = *SCFRDR2;
        *SCFSR2=~RDF;
        return 1;
    } else {
        return 0;
    }

} /* SerialInputByte */


/*
 * read a string with maximum length len from the serial port
 * using a timeout of timeout miliseconds
 *
 * len is the length of array s _including_ the trailing zero,
 * the function returns the number of bytes read _excluding_
 * the trailing zero
 */
int  SerialInputString(char *s, const int len, const int timeout)
{
	char c;
	int i;
	int numRead;
	int skipNewline = 1;
	int maxRead = len - 1;
	
	for(numRead = 0, i = 0; numRead < maxRead;) {
		/* try to get a byte from the serial port */
		while(!SerialInputByte(&c)) {
		}

		/* eat newline characters at start of string */
		if((skipNewline == 1) && (c != '\r') && (c != '\n'))
			skipNewline = 0;

		if(skipNewline == 0) {
			if((c == '\r') || (c == '\n')) {
				s[i++] = '\0';
				return(numRead);
			} else {
				s[i++] = c;
				numRead++;
			}
		}
	}

	return(numRead);
}




/* 
 * SerialInputBlock(): almost the same as SerialInputString(), but
 * this one just reads a block of characters without looking at
 * special characters.
 */
int  SerialInputBlock(char *buf, int bufsize, const int timeout)
{
	char c;
	int i, j;
	int numRead;
	int maxRead = bufsize;
	
	j=timeout;
		
	for(numRead = 0, i = 0; numRead < maxRead;) {
		
		while(!SerialInputByte(&c)) {
			udelay(1000);
			j--;
			/* check timeout value */
			if(!j)
				return(numRead);
		}
		
		buf[i++] = c;
		numRead ++;
	}

	return(numRead);
}

#if 1

#define SOH 1
#define EOT 4
#define NAK 21
#define ACK 6
#define CAN 24
#define STX 2
#define WNTC 'C'

#define SERIALTIMEOUT 3000
//#define D(x,y) *b=x; b++; *b=y; b++	
#define D(x,y)

int xmodem(char *buf, int len) {
	int ok, bs, k, docrc;
	unsigned char block, nakc;
	char *oldbuf;
//	unsigned char *b=(unsigned char *)0x89900000;

	oldbuf=buf;
	block=1;
	ok=50;
	k=0;
	nakc=NAK;
	docrc=0;

	udelay(500000);
	SerialOutputByte(nakc); /* start recieve */
	while(ok) {
		unsigned char c1, c2;
		unsigned short crc;
		int i, j;

		j=SerialInputBlock(&c1,1,SERIALTIMEOUT);
		
		if(j==0) goto nak;
		
		nakc=NAK;
		D(0x10, c1);
		if(c1 == EOT) {
			SerialOutputByte(ACK);
			udelay(100000);
			return buf-oldbuf;
		}
		if(c1 != SOH && c1!=STX) goto nak;
		bs= (c1==SOH) ? 128 : 1024;
		
		SerialInputBlock(&c1,1,SERIALTIMEOUT);
		D(0x20,c1);
		if(c1 != block) goto nak;
		
		SerialInputBlock(&c1,1,SERIALTIMEOUT);
		D(0x30,c1);
		if(c1 != (block^0xff)) goto nak;
		c1=0;
		crc=0;
		for(i=0;i<bs;i++) {
			SerialInputBlock(&c2,1,SERIALTIMEOUT);
			D(0x40,c2);
			buf[i]=c2;
			c1+=c2;
			crc=updcrc(c2, crc);
		}
		if(docrc) {
			SerialInputBlock(&c1,1,SERIALTIMEOUT);
			SerialInputBlock(&c2,1,SERIALTIMEOUT);
			if(c1 != (crc>>8) || c2 != (crc&0xff)) {
				goto nak;
			} else {
				goto ack;
			}
		} else {
			SerialInputBlock(&c2,1,SERIALTIMEOUT);
			D(0x50, c2);
			D(0x57, c1);
			if(c2!=c1) {
				goto nak;
			} else {
				goto ack;
			}
		}
nak:
		/* flush input buffer */
		while(SerialInputBlock(&c2,1,100)) {
			D(0x60, c2);
		}
		SerialOutputByte(nakc);
		D(0xdf, nakc);
		ok--;
		continue;

ack:
		D(0xff, ACK);
		SerialOutputByte(ACK);
		block++;
		buf+=bs;
		
	}
	udelay(100000);
	return 0;
}
#endif

#if 0
#define CRC7D 137

int mb(int b, int d) {
    int n, s, t;

    t=1<<15;
    n=b<<8;
    s=d<<8;
    while(n>255) {
        if(n&t) {
            n=n^s;
        }
        t>>=1;
        s>>=1;
    }
    return n;
}

int crc7(unsigned char *buf, int len) {
    int c=0;
    int i;

    for(i=0;i<len;i++) {
        c=mb(c^buf[i], CRC7D);
    }

    return c;
}

#define TIMEOUT 10

#define STP_START 	'A'
#define STP_OK		'B'
#define STP_RESEND	'C'

int writen(int n, unsigned char *buf) {
	int i;
	i=0;
	while(i<n) {
		SerialOutputByte(buf[i]);
		i++;
	}
	return 0;
}

int readn(int n, unsigned char *buf) {
	int i;
	i=0;
	while(i<n) {
		SerialInputBlock(&buf[i], 1, 0);
		i++;
	}
	return n;
}

int receive(unsigned char *buf, int maxsize) {
	unsigned char pkt[1026];
	int i, t, size, rs, l;
	
	i=1;
	t=TIMEOUT;
	while(t && i) {
		int j;
		j=readn(16, pkt);
		if((pkt[0]==STP_START) && (crc7(pkt,15)==pkt[15])) {
			i=0;
			rs=size=pkt[1]+(pkt[2]<<8)+(pkt[3]<<16)+(pkt[4]<<24);
			for (j=0;j<5;j++)pkt[j]=STP_OK;
			writen(5, pkt);
		} else {
			for (j=0;j<5;j++)pkt[j]=STP_RESEND;
			writen(5, pkt);
		}
		t--;
	}

	if(i)return -1;

	if(size>maxsize)return -size;
	
	while(size) {
		l = size>1024 ? 1024 : size;

		t=TIMEOUT;
		i=1;
		while(t && i) {
                    	unsigned char b[1];
			int j, c;
			readn(l, buf);
			c=crc7(buf, l);
			readn(1, b);
			if(b[0]==c) {
				for (j=0;j<5;j++)pkt[j]=STP_OK;
				writen(5, pkt);
				i=0;
			} else {
				for (j=0;j<5;j++)pkt[j]=STP_RESEND;
				writen(5, pkt);
			}
			t--;
		}

		if(t==0) return -2;
		
		size-=l;
		buf+=l;
	}
	
	return rs;
	
}
#endif

