/*-------------------------------------------------------------------------
 * Filename:      main.c
 * Version:       $Id: main.c,v 1.26 2000/07/10 14:20:17 erikm Exp $
 * Copyright:     Copyright (C) 1999, Jan-Derk Bakker
 * Author:        Jan-Derk Bakker <J.D.Bakker@its.tudelft.nl>
 * Description:   Main file for the trivially simple bootloader for the 
 *                LART boards
 * Created at:    Mon Aug 23 20:00:00 1999
 * Modified by:   Erik Mouw <J.A.K.Mouw@its.tudelft.nl>
 * Modified at:   Sat Mar 25 14:31:16 2000
 *-----------------------------------------------------------------------*/
/*
 * main.c: main file for the blob bootloader
 *
 * Copyright (C) 1999  Jan-Derk Bakker (J.D.Bakker@its.tudelft.nl) and
 *                     Erik Mouw (J.A.K.Mouw@its.tudelft.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
# include "config.h"

#include "clock.h"
#include "command.h"
#include "flash.h"
#include "serial.h"
#include "time.h"
#include "main.h"
#include "util.h"
#include "uucodec.h"
#include "yaffs.h"
#include "nand.h"
#include "mmc.h"
#include "memcommands.h"

void BootKernel(char *commandline);
void Download(char *commandline);
void ResetTerminal(void);
void Reload(char *commandline);
void ReloadMmc(unsigned char *, int, int);

void bra (char *commandline);

//static volatile unsigned short *SCSPTR2 = (unsigned short *)0xffe80020;

void dns(char *commandline);

int nand_startpage, nand_endpage;

int pc(void);

void c_main()
{
    int ker_s=0, ker_l=0, rd_s=0, rd_l=0;
	int numRead = 0;
	char commandline[128], input[64];
	int i, j, button;
	int retval = 0;
	int nandsize, norsize;

	button=0;

	/* Load kernel and ramdisk from flash to RAM */
	SerialOutputString("\r\nMV " BLOB_VERSION "\r\n");

#if 0
	button |= !(*SCSPTR2 & 0x10);
        
    if(button) { 
        SerialOutputString("button pressed\r\n");
		button=1;
    }
#endif

    set_sci();
    udelay(5);
	j=reset_card(0);
	
	for(i = 0; i < 50; i++) {
		SerialOutputByte('.');

 		retval = SerialInputBlock(input, 1, 100); 
		
		if(retval > 0)
			break;
	}
	SerialOutputString("\r\n");
    
	if(retval && input[0]=='Q') goto prompt;

	/* Find type of NAND */
	nandsize=0;
	i=nand_read_id();
	i&=0xffff;
	if(i==0xec75)nandsize=65536; else if(i==0xec76)nandsize=131072;
	if(nandsize==0x20000) {
		nand_startpage=0x19400;
		nand_endpage  =0x1d400;
		if(!MyStrNCmp((char *)0xa0010000, "LongButton", 10)) {
			SerialOutputString("Serial Rescue mode.\r\n");
			nand_startpage=0x0400;
			nand_endpage  =0x4400;
		}
	}else {
		nand_startpage=0xc000;
		nand_endpage=0xfc00;
		if(!MyStrNCmp((char *)0xa0010000, "LongButton", 10)) {
			SerialOutputString("Serial Rescue mode.\r\n");
			nand_startpage=0x0400;
			nand_endpage  =0x3c00;
		}
	}

	/* Find type of NOR */
	if(*(unsigned int *)0xa0000000 == *(unsigned int *)0xa0080000)
		norsize=0x80000; else norsize=0x200000;

	ker_s=0;
	commandline[0]=0;
	
	if(!j) {
		unsigned char *f=(unsigned char *)0x0d000000;
		SerialOutputString("\r\nMMC\r\n");
		read_block(0, f, 0);
		if(!MyStrNCmp("PR000", f, 5) || button) {
			ker_s=MMC_KERNEL_SPEC;
			ker_l=MMC_KERNEL_LEN;
			rd_s=MMC_RD_SPEC;
			rd_l=MMC_RD_LEN;
		} else if(!MyStrNCmp("FirsT", f, 5)) {
			ker_s=24832;
			ker_l=0xc0000>>9;
			rd_s=28032;
			rd_l=0x130000>>9;
		} else if(!MyStrNCmp("Boot ", f, 5)) {
			ker_s=MMC_KERNEL_START;
			ker_l=MMC_KERNEL_LEN;
			rd_s=MMC_RD_START;
			rd_l=MMC_RD_LEN;
		} else if(!MyStrNCmp("PBoot", f, 5)) {
			MyStrNCmp(commandline, f+6, 127);
			ker_s=MMC_KERNEL_START;
			ker_l=MMC_KERNEL_LEN;
			rd_s=MMC_RD_START;
			rd_l=MMC_RD_LEN;
		}
		if(ker_l) {
			f[5]=0;
			SerialOutputString(f);
			SerialOutputString("\r\nKernel ");
			ReloadMmc((unsigned char *)KERNEL_RAM_BASE, ker_l, ker_s); 
			SerialOutputString("\r\nRamdisk ");
			ReloadMmc((unsigned char *)RAMDISK_RAM_BASE, rd_l, rd_s);
			SerialOutputString("\r\n");
		} else if(norsize==0x200000){
			Reload("kernel");
			Reload("ramdisk");
		}
	} else if(norsize==0x200000){
		Reload("kernel");
		Reload("ramdisk");
	}

	SerialOutputString("\r\n");
	
	for(i = 0; i < 1; i++) {
		SerialOutputByte('.');
 		retval = SerialInputBlock(input, 1, 100); 
		if(retval > 0)
			break;
	}
	SerialOutputString("\r\n");
	
	/* if no key was pressed, so proceed booting the kernel */
	if(retval == 0 || input[0]!='Q') {
		if(!ker_s && (norsize==0x80000)) {
			/* if kernel was not on mmc and not on nor, */
			if(nandsize) {
				load_yaffs(0, nand_startpage, nand_endpage);
				MyMemCpyChar(commandline, " boottype-nand console=ttySC1,115200", 38);
			}
		}
		if(commandline[0])BootKernel(commandline); else BootKernel("");
	}

prompt:
	/* the command loop. endless, of course */
	for(;;) {
		DisplayPrompt(NULL);

		/* wait an hour to get a command */
		numRead = GetCommand(commandline, 128, 3600);

		if(numRead > 0) {
			if       (MyStrNCmp(commandline, "boot", 4) == 0) {
								BootKernel(commandline + 4);
			} else if(MyStrNCmp(commandline, "bra ", 4) == 0) {
                            	bra(commandline+3);
			
			} else if(MyStrNCmp(commandline, "pc", 3) == 0) {
                            	SerialOutputHex(pc());
								SerialOutputString("\r\n");

			/* Serial commands */
			} else if(MyStrNCmp(commandline, "reset", 5) == 0) {
				ResetTerminal();
			} else if(MyStrNCmp(commandline, "download ", 9) == 0) {
				Download(commandline + 9);
#if 0
			} else if(MyStrNCmp(commandline, "rz ", 3) == 0) {
				wcreceive((unsigned char *)0x89000000);
#endif
			} else if(MyStrNCmp(commandline, "rxk", 3) == 0) {
                            	xmodem((unsigned char *)KERNEL_RAM_BASE, KERNEL_LEN);
			} else if(MyStrNCmp(commandline, "rxr", 3) == 0) {
                            	xmodem((unsigned char *)RAMDISK_RAM_BASE, RAMDISK_LEN);

			/* nand (yaffs) commands */
			} else if(MyStrNCmp(commandline, "dns ", 4) == 0) {
				dns(commandline+4);
			} else if(MyStrNCmp(commandline, "ypart ", 6) == 0) {
				char *c;
				c=GetHexValue(commandline+6,(unsigned int *)&nand_startpage);
				GetHexValue(c,(unsigned int *)&nand_endpage);

			} else if(MyStrNCmp(commandline, "yload", 5) == 0) {
				load_yaffs(0, nand_startpage, nand_endpage);
			} else if(MyStrNCmp(commandline, "yaffsload", 9) == 0) {
				load_yaffs(1, nand_startpage, nand_endpage);
			} else if(MyStrNCmp(commandline, "yboot", 5) == 0) {
				load_yaffs(0, nand_startpage, nand_endpage);
				BootKernel(" boottype-nand console=ttySC1,115200");
			
			/* ROM commands */
			} else if(MyStrNCmp(commandline, "reload ", 7) == 0) {
				Reload(commandline + 7);
			
			/* MMC commands */
			} else if(MyStrNCmp(commandline, "rmk", 3) == 0) {
				SerialOutputString("\r\nKernel ");
                ReloadMmc((unsigned char *)KERNEL_RAM_BASE, 2048, 60672); 
                SerialOutputString("\r\n");
			} else if(MyStrNCmp(commandline, "rmr", 3) == 0) {
                SerialOutputString("\r\nRamdisk ");
                ReloadMmc((unsigned char *)RAMDISK_RAM_BASE, 12288, 48384); 
                SerialOutputString("\r\n");
			} else if(MyStrNCmp(commandline, "rsk", 3) == 0) {
				SerialOutputString("\r\nSpecial Kernel ");
                ReloadMmc((unsigned char *)KERNEL_RAM_BASE, 2048, 10272); 
                SerialOutputString("\r\n");
			} else if(MyStrNCmp(commandline, "rsr", 3) == 0) {
                SerialOutputString("\r\nSpecial Ramdisk ");
                ReloadMmc((unsigned char *)RAMDISK_RAM_BASE, 10240, 32); 
                SerialOutputString("\r\n");

			/* mem display/set commands */
			} else if(MyStrNCmp(commandline, "dmb ", 4) == 0) {
				dmb(commandline+3);
			} else if(MyStrNCmp(commandline, "dmw ", 4) == 0) {
				dmw(commandline+3);
			} else if(MyStrNCmp(commandline, "dml ", 4) == 0) {
				dml(commandline+3);
			} else if(MyStrNCmp(commandline, "mmb ", 4) == 0) {
              	mmb(commandline+3);
			} else if(MyStrNCmp(commandline, "mmw ", 4) == 0) {
				mmw(commandline+3);
			} else if(MyStrNCmp(commandline, "mml ", 4) == 0) {
				mml(commandline+3);
			} else if(MyStrNCmp(commandline, "smb ", 4) == 0) {
              	smb(commandline+3);
			} else if(MyStrNCmp(commandline, "smw ", 4) == 0) {
				smw(commandline+3);
			} else if(MyStrNCmp(commandline, "sml ", 4) == 0) {
                            	sml(commandline+3);

			} else if(MyStrNCmp(commandline, "timecal", 7) == 0) {
				SerialOutputString("Start\r\n");
				udelay(10000000);
				SerialOutputString("Stop 10s.\r\n");
			
			} else {
				SerialOutputString("?: ");
				SerialOutputString(commandline);
				SerialOutputString("\r\n");
			}
		}
	}
} /* c_main */


void BootKernel(char *commandline)
{
    	unsigned char *param;
        unsigned int *p;
		void (*theKernel)(int zero, int arch) = (void (*)(int, int))KERNEL_RAM_BASE;
		char c;
		volatile unsigned short *SCLSR2 = (unsigned short *)0xffe80024;

        param=(unsigned char *)0xa8001000;
        p=(unsigned int *)param;
        p[0]=1;
        p[1]=0;
        p[2]=0x100;
        p[3]=3;
        p[4]=RAMDISK_RAM_BASE-0x88000000;
        p[5]=0x600000;
        p[6]=1;
        p[7]=1;
        p[8]=1;

        if(*commandline == 0) {
			MyMemCpyChar(param+256, "console=ttySC1,115200", 30);
		} else {
			MyMemCpyChar(param+256, commandline+1, 256 );
		}
        
	while(SerialInputByte(&c)) {
	
	}

	// Clear serial error conditions that make the serial driver hang at
	// startup
	*SCLSR2 = *SCLSR2 & 0xff00;

	/* we assume that the kernel is in place */
	/* See linux/include/asm-arm/system.h for architecture numbers */
	SerialOutputString("\r\nStarting kernel ...\r\n");
	
	theKernel(0,18);

}

void Download(char *commandline)
{
	u32 startAddress = 0;
	int bufLen;
	int numRead = 0;

	if(MyStrNCmp(commandline, "kernel", 6) == 0) {
		/* download kernel */
		startAddress = KERNEL_RAM_BASE;
		bufLen = KERNEL_LEN;
	} else if(MyStrNCmp(commandline, "ramdisk", 7) == 0) {
		/* download ramdisk */
		startAddress = RAMDISK_RAM_BASE;
		bufLen = RAMDISK_LEN;
	} else {
		SerialOutputString("??\"");
		SerialOutputString(commandline);
		SerialOutputString("\"\r\n");
		return;
	}

	numRead = UUDecode((char *)startAddress, bufLen);
	
	if(numRead < 0) {
		/* something went wrong */
		SerialOutputString("\r\nBad download\r\n");
		
		return;
	}
	SerialOutputString("Received ");
	SerialOutputDec(numRead);
	SerialOutputString("\r\n");
}

void ResetTerminal(void)
{
	int i;

	SerialOutputString("          c");
	for(i = 0; i < 100; i++)
		SerialOutputString("\r\n");

	SerialOutputString("c");
}

void ReloadMmc(unsigned char *buf, int blocknum, int block) {
	while(blocknum>0){
		read_block(block, buf, 0);
		block++;
		buf+=512;
		blocknum--;
		if(!(blocknum&15))SerialOutputByte('.');
	}
}


void Reload(char *commandline)
{
	u32 *src = 0;
	u32 *dst = 0;
	int numWords;

	if(MyStrNCmp(commandline, "kernel", 6) == 0) {
            	src = (u32 *)KERNEL_RAM_BASE;
		dst = (u32 *)FLASH_KERNEL_START;
		numWords = FLASH_KERNEL_LEN / 4;
		SerialOutputString("Loading kernel ");
	} else if(MyStrNCmp(commandline, "ramdisk", 7) == 0) {
		src = (u32 *)RAMDISK_RAM_BASE;
		dst = (u32 *)FLASH_INITRD_START;
		numWords = FLASH_INITRD_LEN / 4;
		SerialOutputString("Loading ramdisk ");
	} else {
		SerialOutputString("*( \"");
		SerialOutputString(commandline);
		SerialOutputString("\"\r\n");
		return;
	}

	MyMemCpy(src, dst, numWords);
	SerialOutputString("=\r\n");
}

void bra (char *commandline) {
    unsigned int ad;
    void (*branch)(void);
    
    GetHexValue(commandline,&ad);
    
    branch=(void (*))ad;
    branch();
    
}
void dns(char *commandline) {
	unsigned int ad;
	int i,j;
	unsigned char buffer[528];
	int nandsize;

	/* Find type of NAND */
	nandsize=0;
	i=nand_read_id();
	i&=0xffff;
	if(i==0xec75)nandsize=65536; else if(i==0xec76)nandsize=131072;
	SerialOutputHexNib(i, 4);
	SerialOutputString("\r\n");

	GetHexValue(commandline,&ad);
	nand_read(ad, buffer, nandsize);
	nand_read_oob(ad, buffer+512, nandsize);

	for(i=0;i<528;i+=16) {
		SerialOutputHexNib(i, 3);
		SerialOutputString("  ");
		for(j=0; j<16;j++) {
			SerialOutputHexNib(buffer[i+j], 2);
			SerialOutputString(" ");
		}
		SerialOutputString("  ");
		for(j=0; j<16;j++) {
			if(buffer[i+j]>31) 
				SerialOutputByte(buffer[i+j]); else
				SerialOutputString(" ");
		}
		SerialOutputString("\r\n");
	}
}

int pc() {
	register int r;
	asm("aaa: mova aaa, %0":"=r"(r));
	return r;
}
