/*
 * nldrdefs.h
 *
 * DSP-BIOS Bridge driver support functions for TI OMAP processors.
 *
 * Global Dynamic + static/overlay Node loader (NLDR) constants and types.
 *
 * Copyright (C) 2008 Texas Instruments, Inc.
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef NLDRDEFS_
#define NLDRDEFS_

#include <dspbridge/dbdcddef.h>
#include <dspbridge/devdefs.h>

#define NLDR_MAXPATHLENGTH       255
/* NLDR Objects: */
struct nldr_object;
struct nldr_nodeobject;

/*
 *  ======== nldr_loadtype ========
 *  Load types for a node. Must match values in node.h55.
 */
enum nldr_loadtype {
	NLDR_STATICLOAD,	/* Linked in base image, not overlay */
	NLDR_DYNAMICLOAD,	/* Dynamically loaded node */
	NLDR_OVLYLOAD		/* Linked in base image, overlay node */
};

/*
 *  ======== nldr_ovlyfxn ========
 *  Causes code or data to be copied from load address to run address. This
 *  is the "cod_writefxn" that gets passed to the DBLL_Library and is used as
 *  the ZL write function.
 *
 *  Parameters:
 *      priv_ref:       Handle to identify the node.
 *      ulDspRunAddr:   Run address of code or data.
 *      ulDspLoadAddr:  Load address of code or data.
 *      ul_num_bytes:     Number of (GPP) bytes to copy.
 *      nMemSpace:      RMS_CODE or RMS_DATA.
 *  Returns:
 *      ul_num_bytes:     Success.
 *      0:              Failure.
 *  Requires:
 *  Ensures:
 */
typedef u32(*nldr_ovlyfxn) (void *priv_ref, u32 ulDspRunAddr,
			    u32 ulDspLoadAddr, u32 ul_num_bytes, u32 nMemSpace);

/*
 *  ======== nldr_writefxn ========
 *  Write memory function. Used for dynamic load writes.
 *  Parameters:
 *      priv_ref:       Handle to identify the node.
 *      ulDspAddr:      Address of code or data.
 *      pbuf:           Code or data to be written
 *      ul_num_bytes:     Number of (GPP) bytes to write.
 *      nMemSpace:      DBLL_DATA or DBLL_CODE.
 *  Returns:
 *      ul_num_bytes:     Success.
 *      0:              Failure.
 *  Requires:
 *  Ensures:
 */
typedef u32(*nldr_writefxn) (void *priv_ref,
			     u32 ulDspAddr, void *pbuf,
			     u32 ul_num_bytes, u32 nMemSpace);

/*
 *  ======== nldr_attrs ========
 *  Attributes passed to nldr_create function.
 */
struct nldr_attrs {
	nldr_ovlyfxn pfn_ovly;
	nldr_writefxn pfn_write;
	u16 us_dsp_word_size;
	u16 us_dsp_mau_size;
};

/*
 *  ======== nldr_phase ========
 *  Indicates node create, delete, or execute phase function.
 */
enum nldr_phase {
	NLDR_CREATE,
	NLDR_DELETE,
	NLDR_EXECUTE,
	NLDR_NOPHASE
};

/*
 *  Typedefs of loader functions imported from a DLL, or defined in a
 *  function table.
 */

/*
 *  ======== nldr_allocate ========
 *  Allocate resources to manage the loading of a node on the DSP.
 *
 *  Parameters:
 *      nldr_obj:          Handle of loader that will load the node.
 *      priv_ref:       Handle to identify the node.
 *      node_props:     Pointer to a dcd_nodeprops for the node.
 *      phNldrNode:     Location to store node handle on output. This handle
 *                      will be passed to nldr_load/nldr_unload.
 *      pf_phase_split:   pointer to boolean variable referenced in node.c
 *  Returns:
 *      DSP_SOK:        Success.
 *      DSP_EMEMORY:    Insufficient memory on GPP.
 *  Requires:
 *      nldr_init(void) called.
 *      Valid nldr_obj.
 *      node_props != NULL.
 *      phNldrNode != NULL.
 *  Ensures:
 *      DSP_SOK:        IsValidNode(*phNldrNode).
 *      error:          *phNldrNode == NULL.
 */
typedef dsp_status(*nldr_allocatefxn) (struct nldr_object *nldr_obj,
				       void *priv_ref,
				       IN CONST struct dcd_nodeprops
				       * node_props,
				       OUT struct nldr_nodeobject
				       **phNldrNode,
				       OUT bool *pf_phase_split);

/*
 *  ======== nldr_create ========
 *  Create a loader object. This object handles the loading and unloading of
 *  create, delete, and execute phase functions of nodes on the DSP target.
 *
 *  Parameters:
 *      phNldr:         Location to store loader handle on output.
 *      hdev_obj:     Device for this processor.
 *      pattrs:         Loader attributes.
 *  Returns:
 *      DSP_SOK:        Success;
 *      DSP_EMEMORY:    Insufficient memory for requested resources.
 *  Requires:
 *      nldr_init(void) called.
 *      phNldr != NULL.
 *      hdev_obj != NULL.
 *	pattrs != NULL.
 *  Ensures:
 *      DSP_SOK:        Valid *phNldr.
 *      error:          *phNldr == NULL.
 */
typedef dsp_status(*nldr_createfxn) (OUT struct nldr_object **phNldr,
				     struct dev_object *hdev_obj,
				     IN CONST struct nldr_attrs *pattrs);

/*
 *  ======== nldr_delete ========
 *  Delete the NLDR loader.
 *
 *  Parameters:
 *      nldr_obj:          Node manager object.
 *  Returns:
 *  Requires:
 *      nldr_init(void) called.
 *      Valid nldr_obj.
 *  Ensures:
 *	nldr_obj invalid
 */
typedef void (*nldr_deletefxn) (struct nldr_object *nldr_obj);

/*
 *  ======== nldr_exit ========
 *  Discontinue usage of NLDR module.
 *
 *  Parameters:
 *  Returns:
 *  Requires:
 *      nldr_init(void) successfully called before.
 *  Ensures:
 *      Any resources acquired in nldr_init(void) will be freed when last NLDR
 *      client calls nldr_exit(void).
 */
typedef void (*nldr_exitfxn) (void);

/*
 *  ======== NLDR_Free ========
 *  Free resources allocated in nldr_allocate.
 *
 *  Parameters:
 *      nldr_node_obj:      Handle returned from nldr_allocate().
 *  Returns:
 *  Requires:
 *      nldr_init(void) called.
 *      Valid nldr_node_obj.
 *  Ensures:
 */
typedef void (*nldr_freefxn) (struct nldr_nodeobject *nldr_node_obj);

/*
 *  ======== nldr_get_fxn_addr ========
 *  Get address of create, delete, or execute phase function of a node on
 *  the DSP.
 *
 *  Parameters:
 *      nldr_node_obj:      Handle returned from nldr_allocate().
 *      pstrFxn:        Name of function.
 *      pulAddr:        Location to store function address.
 *  Returns:
 *      DSP_SOK:        Success.
 *      DSP_ESYMBOL:    Address of function not found.
 *  Requires:
 *      nldr_init(void) called.
 *      Valid nldr_node_obj.
 *      pulAddr != NULL;
 *      pstrFxn != NULL;
 *  Ensures:
 */
typedef dsp_status(*nldr_getfxnaddrfxn) (struct nldr_nodeobject
					 * nldr_node_obj,
					 char *pstrFxn, u32 * pulAddr);

/*
 *  ======== nldr_init ========
 *  Initialize the NLDR module.
 *
 *  Parameters:
 *  Returns:
 *      TRUE if initialization succeeded, FALSE otherwise.
 *  Ensures:
 */
typedef bool(*nldr_initfxn) (void);

/*
 *  ======== nldr_load ========
 *  Load create, delete, or execute phase function of a node on the DSP.
 *
 *  Parameters:
 *      nldr_node_obj:      Handle returned from nldr_allocate().
 *      phase:          Type of function to load (create, delete, or execute).
 *  Returns:
 *      DSP_SOK:                Success.
 *      DSP_EMEMORY:            Insufficient memory on GPP.
 *      DSP_EOVERLAYMEMORY:     Can't overlay phase because overlay memory
 *                              is already in use.
 *      DSP_EDYNLOAD:           Failure in dynamic loader library.
 *      DSP_EFWRITE:            Failed to write phase's code or date to target.
 *  Requires:
 *      nldr_init(void) called.
 *      Valid nldr_node_obj.
 *  Ensures:
 */
typedef dsp_status(*nldr_loadfxn) (struct nldr_nodeobject *nldr_node_obj,
				   enum nldr_phase phase);

/*
 *  ======== nldr_unload ========
 *  Unload create, delete, or execute phase function of a node on the DSP.
 *
 *  Parameters:
 *      nldr_node_obj:      Handle returned from nldr_allocate().
 *      phase:          Node function to unload (create, delete, or execute).
 *  Returns:
 *      DSP_SOK:        Success.
 *      DSP_EMEMORY:    Insufficient memory on GPP.
 *  Requires:
 *      nldr_init(void) called.
 *      Valid nldr_node_obj.
 *  Ensures:
 */
typedef dsp_status(*nldr_unloadfxn) (struct nldr_nodeobject *nldr_node_obj,
				     enum nldr_phase phase);

/*
 *  ======== node_ldr_fxns ========
 */
struct node_ldr_fxns {
	nldr_allocatefxn pfn_allocate;
	nldr_createfxn pfn_create;
	nldr_deletefxn pfn_delete;
	nldr_exitfxn pfn_exit;
	nldr_getfxnaddrfxn pfn_get_fxn_addr;
	nldr_initfxn pfn_init;
	nldr_loadfxn pfn_load;
	nldr_unloadfxn pfn_unload;
};

#endif /* NLDRDEFS_ */
