/*
 * OMAP3 Power Management Routines
 *
 * Copyright (C) 2006-2008 Nokia Corporation
 * Tony Lindgren <tony@atomide.com>
 * Jouni Hogander
 *
 * Copyright (C) 2007 Texas Instruments, Inc.
 * Rajendra Nayak <rnayak@ti.com>
 *
 * Copyright (C) 2005 Texas Instruments, Inc.
 * Richard Woodruff <r-woodruff2@ti.com>
 *
 * Based on pm.c for omap1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/pm.h>
#include <linux/suspend.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/list.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/reboot.h>
#include <linux/memblock.h>
#include <linux/tick.h>

#include <plat/sram.h>
#include <plat/clockdomain.h>
#include <plat/powerdomain.h>
#include <plat/resource.h>
#include <plat/control.h>
#include <plat/serial.h>
#include <plat/sdrc.h>
#include <plat/prcm.h>
#include <plat/gpmc.h>
#include <plat/dma.h>
#include <plat/dmtimer.h>
#include <plat/usb.h>

#include <plat/resource.h>

#include <asm/tlbflush.h>

#include "cm.h"
#include "cm-regbits-34xx.h"
#include "prm-regbits-34xx.h"

#include "smartreflex.h"
#include "voltage.h"
#include "prm.h"
#include "pm.h"
#include "sdrc.h"
#include "omap3-opp.h"
#include "sec_hal.h"

static int regset_save_on_suspend;

/* Scratchpad offsets */
#define OMAP343X_TABLE_ADDRESS_OFFSET	   0x31
#define OMAP343X_TABLE_VALUE_OFFSET	   0x30
#define OMAP343X_CONTROL_REG_VALUE_OFFSET  0x32

u32 enable_off_mode = 1;
u32 sleep_while_idle = 1;
u32 wakeup_timer_seconds;
u32 wakeup_timer_milliseconds;
u32 voltage_off_while_idle = 1;
u32 disable_core_off;

#define RESUME_ACTION_REBOOT	1
#define RESUME_ACTION_POWEROFF	2
u32 resume_action;

/* Secure SRAM copy has 0x40 byte header prefixed */
#define OMAP_SECURE_COPY_AREA_SIZE     (0xc000 + 0x40)

struct power_state {
	struct powerdomain *pwrdm;
	u32 next_state;
#ifdef CONFIG_SUSPEND
	u32 saved_state;
#endif
	struct list_head node;
};

static LIST_HEAD(pwrst_list);
static struct power_state *core_pwrst;

static void (*_omap_sram_idle)(u32 *addr, int save_state);

static int secure_ram_dirty;

static int (*_omap_save_secure_sram)(u32 *addr);

static void omap3_pm_mark_secure_ram_dirty(void);

static struct sec_hal_operations pm_sec_hal_ops = {
	.mark_dirty	= omap3_pm_mark_secure_ram_dirty,
};

static unsigned int *_sdrc_counters;
static unsigned int save_sdrc_counters[2];

static struct powerdomain *mpu_pwrdm, *neon_pwrdm;
static struct powerdomain *core_pwrdm, *per_pwrdm, *wkup_pwrdm;
static struct clockdomain *emu_clkdm;
static int secure_ram_save_status;

static struct omap_opp *min_opp, *max_opp;

#define PER_WAKEUP_ERRATA_i582	(1 << 0)
#define RTA_ERRATA_i608		(1 << 1)
#define GPIO_RANDOM_ERRATA_i467	(1 << 2)
#define GPIO_GLITCH_ERRATA_i468	(1 << 3)

#define PER_WAKEUP_DEP_ERRATAS	(PER_WAKEUP_ERRATA_i582 | \
				GPIO_RANDOM_ERRATA_i467 | \
				GPIO_GLITCH_ERRATA_i468)

static u16 pm34xx_errata;
#define IS_PM34XX_ERRATA(id)	(pm34xx_errata & (id))

static s64 next_wdt_expire;

void pm_wdt_ping(unsigned timeout)
{
	next_wdt_expire = sched_clock() + timeout * (s64)NSEC_PER_SEC;
}

static inline void omap3_per_save_context(void)
{
	omap_gpio_save_context();
}

static inline void omap3_per_restore_context(void)
{
	omap_gpio_restore_context();
}

static void omap3_enable_io_chain(void)
{
	int timeout = 0;

	if (omap_rev() >= OMAP3430_REV_ES3_1) {
		prm_set_mod_reg_bits(OMAP3430_EN_IO_CHAIN, WKUP_MOD, PM_WKEN);
		/* Do a readback to assure write has been done */
		prm_read_mod_reg(WKUP_MOD, PM_WKEN);

		while (!(prm_read_mod_reg(WKUP_MOD, PM_WKST) &
			 OMAP3430_ST_IO_CHAIN)) {
			timeout++;
			if (timeout > 1000) {
				printk(KERN_ERR "Wake up daisy chain "
				       "activation failed.\n");
				return;
			}
			prm_set_mod_reg_bits(OMAP3430_ST_IO_CHAIN,
					     WKUP_MOD, PM_WKST);
		}
	}
}

static void omap3_disable_io_chain(void)
{
	if (omap_rev() >= OMAP3430_REV_ES3_1)
		prm_clear_mod_reg_bits(OMAP3430_EN_IO_CHAIN, WKUP_MOD, PM_WKEN);
}

static void omap3_core_save_context(void)
{
	u32 control_padconf_off;

	/* Save the padconf registers */
	control_padconf_off = omap_ctrl_readl(OMAP343X_CONTROL_PADCONF_OFF);
	control_padconf_off |= START_PADCONF_SAVE;
	omap_ctrl_writel(control_padconf_off, OMAP343X_CONTROL_PADCONF_OFF);
	/* wait for the save to complete */
	while (!(omap_ctrl_readl(OMAP343X_CONTROL_GENERAL_PURPOSE_STATUS)
			& PADCONF_SAVE_DONE))
		udelay(1);

	/*
	 * Force write last pad into memory, as this can fail in some
	 * cases according to erratas 1.157, 1.185
	 */
	omap_ctrl_writel(omap_ctrl_readl(OMAP343X_PADCONF_ETK_D14),
		OMAP343X_CONTROL_MEM_WKUP + 0x2a0);

	/* Save the Interrupt controller context */
	omap_intc_save_context();
	/* Save the GPMC context */
	omap3_gpmc_save_context();
	/* Save the system control module context, padconf already save above*/
	omap3_control_save_context();
	omap_dma_global_context_save();
}

static void omap3_core_restore_context(void)
{
	/* Restore the control module context, padconf restored by h/w */
	omap3_control_restore_context();
	/* Restore the GPMC context */
	omap3_gpmc_restore_context();
	/* Restore the interrupt controller context */
	omap_intc_restore_context();
	omap_dma_global_context_restore();
}

/*
 * FIXME: This function should be called before entering off-mode after
 * OMAP3 secure services have been accessed. Currently it is only called
 * once during boot sequence, but this works as we are not using secure
 * services.
 */
static void omap3_save_secure_ram_context(u32 target_mpu_state)
{
	if (omap_type() != OMAP2_DEVICE_TYPE_GP) {
		/*
		 * MPU next state must be set to POWER_ON temporarily,
		 * otherwise the WFI executed inside the ROM code
		 * will hang the system.
		 */
		omap3_pwrdm_set_next_pwrst(mpu_pwrdm, PWRDM_POWER_ON);
		secure_ram_save_status = _omap_save_secure_sram((u32 *)
				__pa(omap3_secure_ram_storage));
		omap3_pwrdm_set_next_pwrst(mpu_pwrdm, target_mpu_state);
		if (secure_ram_save_status >= 0)
			secure_ram_dirty = false;
	}
}

/*
 * PRCM Interrupt Handler Helper Function
 *
 * The purpose of this function is to clear any wake-up events latched
 * in the PRCM PM_WKST_x registers. It is possible that a wake-up event
 * may occur whilst attempting to clear a PM_WKST_x register and thus
 * set another bit in this register. A while loop is used to ensure
 * that any peripheral wake-up events occurring while attempting to
 * clear the PM_WKST_x are detected and cleared.
 */
static int prcm_clear_mod_irqs(s16 module, u8 regs)
{
	u32 wkst, fclk, iclk, clken;
	u16 wkst_off = (regs == 3) ? OMAP3430ES2_PM_WKST3 : PM_WKST1;
	u16 fclk_off = (regs == 3) ? OMAP3430ES2_CM_FCLKEN3 : CM_FCLKEN1;
	u16 iclk_off = (regs == 3) ? CM_ICLKEN3 : CM_ICLKEN1;
	u16 grpsel_off = (regs == 3) ?
		OMAP3430ES2_PM_MPUGRPSEL3 : OMAP3430_PM_MPUGRPSEL;
	int c = 0;

	wkst = prm_read_mod_reg(module, wkst_off);
	wkst &= prm_read_mod_reg(module, grpsel_off);
	if (wkst) {
		iclk = cm_read_mod_reg(module, iclk_off);
		fclk = cm_read_mod_reg(module, fclk_off);
		while (wkst) {
			clken = wkst;
			cm_set_mod_reg_bits(clken, module, iclk_off);
			/*
			 * For USBHOST, we don't know whether HOST1 or
			 * HOST2 woke us up, so enable both f-clocks
			 */
			if (module == OMAP3430ES2_USBHOST_MOD)
				clken |= 1 << OMAP3430ES2_EN_USBHOST2_SHIFT;
			cm_set_mod_reg_bits(clken, module, fclk_off);
			prm_write_mod_reg(wkst, module, wkst_off);
			wkst = prm_read_mod_reg(module, wkst_off);
			c++;
		}
		cm_write_mod_reg(iclk, module, iclk_off);
		cm_write_mod_reg(fclk, module, fclk_off);
	}

	return c;
}

static int _prcm_int_handle_wakeup(void)
{
	int c;

	c = prcm_clear_mod_irqs(WKUP_MOD, 1);
	c += prcm_clear_mod_irqs(CORE_MOD, 1);
	c += prcm_clear_mod_irqs(OMAP3430_PER_MOD, 1);
	if (omap_rev() > OMAP3430_REV_ES1_0) {
		c += prcm_clear_mod_irqs(CORE_MOD, 3);
		c += prcm_clear_mod_irqs(OMAP3430ES2_USBHOST_MOD, 1);
	}

	return c;
}

/*
 * PRCM Interrupt Handler
 *
 * The PRM_IRQSTATUS_MPU register indicates if there are any pending
 * interrupts from the PRCM for the MPU. These bits must be cleared in
 * order to clear the PRCM interrupt. The PRCM interrupt handler is
 * implemented to simply clear the PRM_IRQSTATUS_MPU in order to clear
 * the PRCM interrupt. Please note that bit 0 of the PRM_IRQSTATUS_MPU
 * register indicates that a wake-up event is pending for the MPU and
 * this bit can only be cleared if the all the wake-up events latched
 * in the various PM_WKST_x registers have been cleared. The interrupt
 * handler is implemented using a do-while loop so that if a wake-up
 * event occurred during the processing of the prcm interrupt handler
 * (setting a bit in the corresponding PM_WKST_x register and thus
 * preventing us from clearing bit 0 of the PRM_IRQSTATUS_MPU register)
 * this would be handled.
 */
static irqreturn_t prcm_interrupt_handler (int irq, void *dev_id)
{
	u32 irqenable_mpu, irqstatus_mpu;
	int c = 0;
	int ct = 0;

	irqenable_mpu = prm_read_mod_reg(OCP_MOD,
					 OMAP3_PRM_IRQENABLE_MPU_OFFSET);
	irqstatus_mpu = prm_read_mod_reg(OCP_MOD,
					 OMAP3_PRM_IRQSTATUS_MPU_OFFSET);
	irqstatus_mpu &= irqenable_mpu;

	do {
		if (irqstatus_mpu & (OMAP3430_WKUP_ST | OMAP3430_IO_ST)) {
			c = _prcm_int_handle_wakeup();
			ct++;

			/*
			 * Is the MPU PRCM interrupt handler racing with the
			 * IVA2 PRCM interrupt handler ?
			 */
			WARN(!c && (ct == 1), "prcm: WARNING: PRCM indicated "
					"MPU wakeup but no wakeup sources "
					"are marked\n");
		} else {
			/* XXX we need to expand our PRCM interrupt handler */
			WARN(1, "prcm: WARNING: PRCM interrupt received, but "
			     "no code to handle it (%08x)\n", irqstatus_mpu);
		}

		prm_write_mod_reg(irqstatus_mpu, OCP_MOD,
					OMAP3_PRM_IRQSTATUS_MPU_OFFSET);

		irqstatus_mpu = prm_read_mod_reg(OCP_MOD,
					OMAP3_PRM_IRQSTATUS_MPU_OFFSET);
		irqstatus_mpu &= irqenable_mpu;

	} while (irqstatus_mpu);

	return IRQ_HANDLED;
}

static void restore_control_register(u32 val)
{
	__asm__ __volatile__ ("mcr p15, 0, %0, c1, c0, 0" : : "r" (val));
}

static inline void omap3_save_neon_context(void)
{
#ifdef CONFIG_VFP
	vfp_pm_save_context();
#endif
}

/* Function to restore the table entry that was modified for enabling MMU */
static void restore_table_entry(void)
{
	u32 *scratchpad_address;
	u32 previous_value, control_reg_value;
	u32 *address;

	scratchpad_address = OMAP2_L4_IO_ADDRESS(OMAP343X_SCRATCHPAD);

	/* Get address of entry that was modified */
	address = (u32 *)__raw_readl(scratchpad_address +
				     OMAP343X_TABLE_ADDRESS_OFFSET);
	/* Get the previous value which needs to be restored */
	previous_value = __raw_readl(scratchpad_address +
				     OMAP343X_TABLE_VALUE_OFFSET);
	address = __va(address);
	*address = previous_value;
	flush_tlb_all();
	control_reg_value = __raw_readl(scratchpad_address
					+ OMAP343X_CONTROL_REG_VALUE_OFFSET);
	/* This will enable caches and prediction */
	restore_control_register(control_reg_value);
}

void omap_sram_idle(void)
{
	/* Variable to tell what needs to be saved and restored
	 * in omap_sram_idle*/
	/* save_state = 0 => Nothing to save and restored */
	/* save_state = 1 => Only L1 and logic lost */
	/* save_state = 2 => Only L2 lost */
	/* save_state = 3 => L1, L2 and logic lost */
	int save_state = 0;
	int mpu_next_state = PWRDM_POWER_ON;
	int per_next_state = PWRDM_POWER_ON;
	int core_next_state = PWRDM_POWER_ON;
	int core_prev_state = PWRDM_POWER_ON;
	int per_prev_state = PWRDM_POWER_ON;
	u32 sdrc_pwr = 0;
	u32 fclk_status = 0;

	if (!_omap_sram_idle)
		return;

	pwrdm_clear_all_prev_pwrst(mpu_pwrdm);
	pwrdm_clear_all_prev_pwrst(neon_pwrdm);
	pwrdm_clear_all_prev_pwrst(core_pwrdm);
	pwrdm_clear_all_prev_pwrst(per_pwrdm);

	mpu_next_state = omap3_pwrdm_read_next_pwrst(mpu_pwrdm);
	switch (mpu_next_state) {
	case PWRDM_POWER_ON:
	case PWRDM_POWER_INACTIVE:
	case PWRDM_POWER_RET:
		/* No need to save context */
		save_state = 0;
		break;
	case PWRDM_POWER_OFF:
		save_state = 3;
		break;
	default:
		/* Invalid state */
		printk(KERN_ERR "Invalid mpu state in sram_idle\n");
		return;
	}

	if (mpu_next_state != PWRDM_POWER_ON) {
		pwrdm_pre_transition(mpu_pwrdm);
		pwrdm_pre_transition(neon_pwrdm);
		pwrdm_pre_transition(core_pwrdm);
		pwrdm_pre_transition(per_pwrdm);
	}

	/* WDT hack */
	core_next_state = omap3_pwrdm_read_next_pwrst(core_pwrdm);
	per_next_state = omap3_pwrdm_read_next_pwrst(per_pwrdm);
	if (mpu_next_state == PWRDM_POWER_OFF && next_wdt_expire) {
		struct timespec t;
		s64 next_wake;
		t = ktime_to_timespec(tick_nohz_get_sleep_length());
		next_wake = sched_clock() + t.tv_sec * (s64)NSEC_PER_SEC +
			t.tv_nsec;
		if (next_wake > next_wdt_expire) {
			pr_info("WDT expire imminent! Preventing off.\n");
			mpu_next_state = PWRDM_POWER_RET;
			if (core_next_state == PWRDM_POWER_OFF)
				core_next_state = PWRDM_POWER_RET;
			if (per_next_state == PWRDM_POWER_OFF)
				per_next_state = PWRDM_POWER_RET;
			omap3_pwrdm_set_next_pwrst(mpu_pwrdm, mpu_next_state);
			omap3_pwrdm_set_next_pwrst(core_pwrdm, core_next_state);
			omap3_pwrdm_set_next_pwrst(per_pwrdm, per_next_state);
			save_state = 0;
		}
	}

	if (core_next_state <= PWRDM_POWER_RET) {
		/*
		 * Check if SmartReflex is happy enough for VDD2.
		 * Otherwise, stop idle path.
		 */
		if (omap_smartreflex_disable(VDD2,
			omap_get_volt_data(VDD2, get_curr_vdd2_voltage()),
			1)) {
			pwrdm_set_next_pwrst(core_pwrdm, PWRDM_POWER_ON);
			core_next_state = PWRDM_POWER_ON;
		}

		program_vdd2_opp_dll_wa(min_opp, max_opp);
		cm_rmw_mod_reg_bits(OMAP3430_AUTO_CORE_DPLL_MASK,
				0x1, PLL_MOD, CM_AUTOIDLE);
		if (core_next_state == PWRDM_POWER_OFF)
			if (IS_PM34XX_ERRATA(PER_WAKEUP_DEP_ERRATAS))
				pwrdm_add_wkdep(per_pwrdm, wkup_pwrdm);
	}


	/* NEON control */
	if (pwrdm_read_pwrst(neon_pwrdm) == PWRDM_POWER_ON) {
		omap3_pwrdm_set_next_pwrst(neon_pwrdm, mpu_next_state);
		if (mpu_next_state == PWRDM_POWER_OFF) {
			omap3_save_neon_context();
			omap3_emu_mpusubsystem_save_context();
		}
	}

	/* PER */
	if (per_next_state < PWRDM_POWER_ON) {
		omap2_gpio_prepare_for_idle(per_next_state);
		if (per_next_state == PWRDM_POWER_OFF)
			omap3_per_save_context();
		omap_uart_prepare_idle(2);
	}

	/*
	 * Disable smartreflex before entering WFI.
	 * Only needed if we are going to enter retention or off.
	 */
	if (mpu_next_state <= PWRDM_POWER_RET) {
		/*
		 * Check if SmartReflex is happy enough with VDD1.
		 * Otherwise, stop idle path.
		 */
		if (omap_smartreflex_disable(VDD1,
			omap_get_volt_data(VDD1, get_curr_vdd1_voltage()),
			1)) {
			pwrdm_set_next_pwrst(mpu_pwrdm, PWRDM_POWER_ON);
			mpu_next_state = PWRDM_POWER_ON;
		}
	}

	/*
	 * We need to disable the autoidle bit from MPU INTC,
	 * otherwise INTC would get stall, and we would never
	 * come out of WFI. This is done here because
	 * save secure ram also does WFI.
	 */
	omap3_intc_prepare_idle();

	/* CORE */
	if (core_next_state < PWRDM_POWER_ON) {
		omap_uart_prepare_idle(0);
		omap_uart_prepare_idle(1);
		omap_voltage_vc_update(core_next_state);
		if (core_next_state == PWRDM_POWER_OFF) {
			omap3_core_save_context();
			omap3_prcm_save_context();
			omap3_save_secure_ram_context(mpu_next_state);
		}
		if (core_next_state <= PWRDM_POWER_RET) {
			/* Enable IO-PAD and IO-CHAIN wakeups */
			prm_set_mod_reg_bits(OMAP3430_EN_IO, WKUP_MOD, PM_WKEN);
			omap3_enable_io_chain();
		}
	}

	/*
	* On EMU/HS devices ROM code restores a SRDC value
	* from scratchpad which has automatic self refresh on timeout
	* of AUTO_CNT = 1 enabled. This takes care of errata 1.142.
	* Hence store/restore the SDRC_POWER register here.
	*/
	if (cpu_is_omap3430() &&
	    omap_rev() >= OMAP3430_REV_ES3_0 &&
	    omap_type() != OMAP2_DEVICE_TYPE_GP &&
	    core_next_state == PWRDM_POWER_OFF)
		sdrc_pwr = sdrc_read_reg(SDRC_POWER);

	if (regset_save_on_suspend)
		pm_dbg_regset_save(1);

	/*
	 * omap3_arm_context is the location where ARM registers
	 * get saved. The restore path then reads from this
	 * location and restores them back.
	 */
	_omap_sram_idle(omap3_arm_context, save_state);
	cpu_init();

	if (regset_save_on_suspend)
		pm_dbg_regset_save(2);

	/* Restore normal SDRC POWER settings */
	if (cpu_is_omap3430() &&
	    omap_rev() >= OMAP3430_REV_ES3_0 &&
	    omap_type() != OMAP2_DEVICE_TYPE_GP &&
	    core_next_state == PWRDM_POWER_OFF)
		sdrc_write_reg(sdrc_pwr, SDRC_POWER);

	/* Restore table entry modified during MMU restoration */
	if (pwrdm_read_prev_pwrst(mpu_pwrdm) == PWRDM_POWER_OFF) {
		restore_table_entry();
		omap3_emu_mpusubsystem_restore_context();
	}

	/* CORE */
	if (core_next_state < PWRDM_POWER_ON) {
		core_prev_state = pwrdm_read_prev_pwrst(core_pwrdm);
		if (core_prev_state == PWRDM_POWER_OFF) {
			omap3_core_restore_context();
			omap3_prcm_restore_context();
			omap3_sram_restore_context();
			omap2_sms_restore_context();
			/*
			 * Errata 1.164 fix : OTG autoidle can prevent
			 * sleep
			 */
			if (cpu_is_omap3430())
				usb_musb_disable_autoidle();
			memcpy(_sdrc_counters, save_sdrc_counters, sizeof(save_sdrc_counters));
		}
		omap_uart_resume_idle(0);
		omap_uart_resume_idle(1);
	}
	omap3_intc_resume_idle();

	memcpy(save_sdrc_counters, _sdrc_counters, sizeof(save_sdrc_counters));

	/*
	 * Enable smartreflex after WFI. Only needed if we entered
	 * retention or off
	 */
	if (mpu_next_state <= PWRDM_POWER_RET)
		omap_smartreflex_enable(VDD1, omap_get_volt_data(VDD1,
					get_curr_vdd1_voltage()));
	if (core_next_state <= PWRDM_POWER_RET) {
		cm_rmw_mod_reg_bits(OMAP3430_AUTO_CORE_DPLL_MASK,
			0x0, PLL_MOD, CM_AUTOIDLE);
		reprogram_vdd2_opp_dll_wa(min_opp, max_opp);
		omap_smartreflex_enable(VDD2, omap_get_volt_data(VDD2,
					get_curr_vdd2_voltage()));
		if (core_next_state == PWRDM_POWER_OFF)
			if (IS_PM34XX_ERRATA(PER_WAKEUP_DEP_ERRATAS))
				pwrdm_del_wkdep(per_pwrdm, wkup_pwrdm);
	}

	/* PER */
	if (per_next_state < PWRDM_POWER_ON) {
		if (per_next_state == PWRDM_POWER_OFF) {
			/*
			 * Reading the prev-state takes long time (11us@OPP2),
			 * only do it, if we really tried to put PER in OFF
			 */
			per_prev_state = pwrdm_read_prev_pwrst(per_pwrdm);
			if (per_prev_state == PWRDM_POWER_OFF) {
				omap3_per_restore_context();
				omap3_gpio_restore_pad_context(0);
			} else if (per_next_state == PWRDM_POWER_OFF) {
				omap3_gpio_restore_pad_context(1);
			}
		}
		omap2_gpio_resume_after_idle();
		omap_uart_resume_idle(2);
		if (IS_PM34XX_ERRATA(PER_WAKEUP_ERRATA_i582) &&
					omap_uart_check_per_uarts_used()) {
			if ((core_prev_state == PWRDM_POWER_ON) &&
					(per_prev_state == PWRDM_POWER_OFF)) {
				pr_err("Corner case of Errata i582 \n");
				/*
				 * We dont seem to have a real recovery
				 * other than reset
				 */
		/*
		 * Errata i582:Alternative available here is to do a reboot OR
		 * go to per off/core off,we will just print and cause uart to
		 * be in an unstable state and continue on till we hit the next
		 * off transition.Reboot of the device due to this corner
		 * case is undesirable
		 */
				if (omap_uart_per_errata())/* warm reset reqd */
					pr_err("%s: PER UART hit with \
					Errata i582 Corner case.\n", __func__);
			}
		}
	}

	/* Disable IO-PAD and IO-CHAIN wakeup */
	if (core_next_state <= PWRDM_POWER_RET) {
		prm_clear_mod_reg_bits(OMAP3430_EN_IO, WKUP_MOD, PM_WKEN);
		omap3_disable_io_chain();
	}

	if (mpu_next_state != PWRDM_POWER_ON) {
		pwrdm_post_transition(mpu_pwrdm);
		pwrdm_post_transition(neon_pwrdm);
		pwrdm_post_transition(core_pwrdm);
		pwrdm_post_transition(per_pwrdm);
	}

	/* Check if secure RAM save failed for some reason */
	if (secure_ram_save_status < 0)
		WARN(1, "secure ram save failed (%d)\n",
			secure_ram_save_status);
	/* TODO: Account number of saved cases: secure_ram_save_status == 0 */
	if (secure_ram_save_status)
		secure_ram_save_status = 0;
}

int omap3_can_sleep(void)
{
	if (!sleep_while_idle)
		return 0;
	if (!omap_uart_can_sleep())
		return 0;
	return 1;
}

struct powerdomain_data {
	u8 next_state;
	u8 init_done;
};

static struct powerdomain_data mpu_pwrdm_data;
static struct powerdomain_data core_pwrdm_data;
static struct powerdomain_data neon_pwrdm_data;

static struct powerdomain_data *get_pwrdm_data(struct powerdomain *pwrdm)
{
	if (pwrdm == mpu_pwrdm)
		return &mpu_pwrdm_data;
	else if (pwrdm == core_pwrdm)
		return &core_pwrdm_data;
	else if (pwrdm == neon_pwrdm)
		return &neon_pwrdm_data;
	return NULL;
}

int omap3_pwrdm_set_next_pwrst(struct powerdomain *pwrdm, u8 pwrst)
{
	struct powerdomain_data *data = get_pwrdm_data(pwrdm);
	u8 prg_pwrst;

	if (!data)
		return pwrdm_set_next_pwrst(pwrdm, pwrst);

	if (!data->init_done)
		data->init_done = 1;
	else if (data->next_state == pwrst)
		return 0;

	if (pwrst == PWRDM_POWER_INACTIVE)
		prg_pwrst = PWRDM_POWER_ON;
	else
		prg_pwrst = pwrst;

	pwrdm_set_next_pwrst(pwrdm, prg_pwrst);

	if (pwrst == PWRDM_POWER_ON)
		omap2_clkdm_deny_idle(pwrdm->pwrdm_clkdms[0]);
	else
		omap2_clkdm_allow_idle(pwrdm->pwrdm_clkdms[0]);

	data->next_state = pwrst;
	return 0;
}

int omap3_pwrdm_read_next_pwrst(struct powerdomain *pwrdm)
{
	struct powerdomain_data *data = get_pwrdm_data(pwrdm);

	if (!data || !data->init_done)
		return pwrdm_read_next_pwrst(pwrdm);
	return data->next_state;
}

/* This sets pwrdm state (other than mpu & core. Currently only ON &
 * RET are supported. Function is assuming that clkdm doesn't have
 * hw_sup mode enabled. */
int set_pwrdm_state(struct powerdomain *pwrdm, u32 state)
{
	u32 cur_state;
	int sleep_switch = 0;
	int ret = 0;

	if (pwrdm == NULL || IS_ERR(pwrdm))
		return -EINVAL;

	if (state != PWRDM_POWER_INACTIVE)
		while (!(pwrdm->pwrsts & (1 << state))) {
			if (state == PWRDM_POWER_OFF)
				return ret;
			state--;
		}

	cur_state = pwrdm_read_next_pwrst(pwrdm);
	if (cur_state == state)
		return ret;

	if (pwrdm_read_pwrst(pwrdm) < PWRDM_POWER_ON) {
		omap2_clkdm_wakeup(pwrdm->pwrdm_clkdms[0]);
		sleep_switch = 1;
		pwrdm_wait_transition(pwrdm);
	}

	ret = omap3_pwrdm_set_next_pwrst(pwrdm, state);
	if (ret) {
		printk(KERN_ERR "Unable to set state of powerdomain: %s\n",
		       pwrdm->name);
		goto err;
	}

	if (sleep_switch) {
		omap2_clkdm_allow_idle(pwrdm->pwrdm_clkdms[0]);
		pwrdm_wait_transition(pwrdm);
		pwrdm_state_switch(pwrdm);
	}

err:
	return ret;
}

/* return a pointer to the sdrc counters */
unsigned int *omap3_get_sdrc_counters(void)
{
	return _sdrc_counters;
}

static void omap3_pm_idle(void)
{
	local_irq_disable();
	local_fiq_disable();

	if (!omap3_can_sleep())
		goto out;

	if (omap_irq_pending() || need_resched())
		goto out;

	omap_sram_idle();

out:
	local_fiq_enable();
	local_irq_enable();
}

#ifdef CONFIG_SUSPEND
static suspend_state_t suspend_state;

static void omap2_pm_wakeup_on_timer(u32 seconds, u32 milliseconds)
{
	u32 tick_rate, cycles;

	if (!seconds && !milliseconds)
		return;

	tick_rate = clk_get_rate(omap_dm_timer_get_fclk(gptimer_wakeup));
	cycles = tick_rate * seconds + tick_rate * milliseconds / 1000;
	omap_dm_timer_stop(gptimer_wakeup);
	omap_dm_timer_set_load_start(gptimer_wakeup, 0, 0xffffffff - cycles);

	pr_info("PM: Resume timer in %u.%03u secs"
		" (%d ticks at %d ticks/sec.)\n",
		seconds, milliseconds, cycles, tick_rate);
}

static int omap3_pm_prepare(void)
{
	disable_hlt();
	return 0;
}

static int omap3_pm_suspend(void)
{
	struct power_state *pwrst;
	int state, ret = 0;

	if (wakeup_timer_seconds || wakeup_timer_milliseconds)
		omap2_pm_wakeup_on_timer(wakeup_timer_seconds,
					 wakeup_timer_milliseconds);

	/* Read current next_pwrsts */
	list_for_each_entry(pwrst, &pwrst_list, node)
		pwrst->saved_state = omap3_pwrdm_read_next_pwrst(pwrst->pwrdm);
	/* Set ones wanted by suspend */
	list_for_each_entry(pwrst, &pwrst_list, node) {
		if (set_pwrdm_state(pwrst->pwrdm, pwrst->next_state))
			goto restore;
		if (pwrdm_clear_all_prev_pwrst(pwrst->pwrdm))
			goto restore;
	}

	omap_uart_prepare_suspend();
	omap3_intc_suspend();

	regset_save_on_suspend = 1;
	omap_sram_idle();
	regset_save_on_suspend = 0;

restore:
	/* Restore next_pwrsts */
	list_for_each_entry(pwrst, &pwrst_list, node) {
		state = pwrdm_read_prev_pwrst(pwrst->pwrdm);
		if (state > pwrst->next_state) {
			printk(KERN_INFO "Powerdomain (%s) didn't enter "
			       "target state %d\n",
			       pwrst->pwrdm->name, pwrst->next_state);
			ret = -1;
		}
		set_pwrdm_state(pwrst->pwrdm, pwrst->saved_state);
	}
	if (ret)
		printk(KERN_ERR "Could not enter target state in pm_suspend\n");
	else
		printk(KERN_INFO "Successfully put all powerdomains "
		       "to target state\n");

	return ret;
}

static int omap3_pm_enter(suspend_state_t unused)
{
	int ret = 0;

	switch (suspend_state) {
	case PM_SUSPEND_STANDBY:
	case PM_SUSPEND_MEM:
		ret = omap3_pm_suspend();
		break;
	default:
		ret = -EINVAL;
	}

	return ret;
}

static void omap3_pm_finish(void)
{
	enable_hlt();

	if (resume_action & RESUME_ACTION_REBOOT) {
		char boot_cmd[2] = "n";

		/* Optional boot mode in bits 16..23 */
		if (resume_action >> 16)
			boot_cmd[0] = (resume_action >> 16) & 0xFF;

		machine_restart(boot_cmd);
	} else if (resume_action & RESUME_ACTION_POWEROFF) {
		BUG_ON(!pm_power_off);
		pm_power_off();
	}
}

/* Hooks to enable / disable UART interrupts during suspend */
static int omap3_pm_begin(suspend_state_t state)
{
	suspend_state = state;
	omap_uart_enable_irqs(0);
	return 0;
}

static void omap3_pm_end(void)
{
	suspend_state = PM_SUSPEND_ON;
	omap_uart_enable_irqs(1);
	return;
}

static struct platform_suspend_ops omap_pm_ops = {
	.begin		= omap3_pm_begin,
	.end		= omap3_pm_end,
	.prepare	= omap3_pm_prepare,
	.enter		= omap3_pm_enter,
	.finish		= omap3_pm_finish,
	.valid		= suspend_valid_only_mem,
};
#endif /* CONFIG_SUSPEND */


/**
 * omap3_iva_idle(): ensure IVA is in idle so it can be put into
 *                   retention
 *
 * In cases where IVA2 is activated by bootcode, it may prevent
 * full-chip retention or off-mode because it is not idle.  This
 * function forces the IVA2 into idle state so it can go
 * into retention/off and thus allow full-chip retention/off.
 *
 **/
static void __init omap3_iva_idle(void)
{
	/* ensure IVA2 clock is disabled */
	cm_write_mod_reg(0, OMAP3430_IVA2_MOD, CM_FCLKEN);

	/* if no clock activity, nothing else to do */
	if (!(cm_read_mod_reg(OMAP3430_IVA2_MOD, OMAP3430_CM_CLKSTST) &
	      OMAP3430_CLKACTIVITY_IVA2_MASK))
		return;

	/* Reset IVA2 */
	prm_write_mod_reg(OMAP3430_RST1_IVA2 |
			  OMAP3430_RST2_IVA2 |
			  OMAP3430_RST3_IVA2,
			  OMAP3430_IVA2_MOD, RM_RSTCTRL);

	/* Enable IVA2 clock */
	cm_write_mod_reg(OMAP3430_CM_FCLKEN_IVA2_EN_IVA2_MASK,
			 OMAP3430_IVA2_MOD, CM_FCLKEN);

	/* Set IVA2 boot mode to 'idle' */
	omap_ctrl_writel(OMAP3_IVA2_BOOTMOD_IDLE,
			 OMAP343X_CONTROL_IVA2_BOOTMOD);

	/* Un-reset IVA2 */
	prm_write_mod_reg(0, OMAP3430_IVA2_MOD, RM_RSTCTRL);

	/* Disable IVA2 clock */
	cm_write_mod_reg(0, OMAP3430_IVA2_MOD, CM_FCLKEN);

	/* Reset IVA2 */
	prm_write_mod_reg(OMAP3430_RST1_IVA2 |
			  OMAP3430_RST2_IVA2 |
			  OMAP3430_RST3_IVA2,
			  OMAP3430_IVA2_MOD, RM_RSTCTRL);
}

static void __init omap3_d2d_idle(void)
{
	u16 mask, padconf;

	/* In a stand alone OMAP3430 where there is not a stacked
	 * modem for the D2D Idle Ack and D2D MStandby must be pulled
	 * high. S CONTROL_PADCONF_SAD2D_IDLEACK and
	 * CONTROL_PADCONF_SAD2D_MSTDBY to have a pull up. */
	mask = (1 << 4) | (1 << 3); /* pull-up, enabled */
	padconf = omap_ctrl_readw(OMAP3_PADCONF_SAD2D_MSTANDBY);
	padconf |= mask;
	omap_ctrl_writew(padconf, OMAP3_PADCONF_SAD2D_MSTANDBY);

	padconf = omap_ctrl_readw(OMAP3_PADCONF_SAD2D_IDLEACK);
	padconf |= mask;
	omap_ctrl_writew(padconf, OMAP3_PADCONF_SAD2D_IDLEACK);

	/* reset modem */
	prm_write_mod_reg(OMAP3430_RM_RSTCTRL_CORE_MODEM_SW_RSTPWRON |
			  OMAP3430_RM_RSTCTRL_CORE_MODEM_SW_RST,
			  CORE_MOD, RM_RSTCTRL);
	prm_write_mod_reg(0, CORE_MOD, RM_RSTCTRL);
}

static void __init prcm_setup_regs(void)
{
	/* XXX Reset all wkdeps. This should be done when initializing
	 * powerdomains */
	prm_write_mod_reg(0, OMAP3430_IVA2_MOD, PM_WKDEP);
	prm_write_mod_reg(0, MPU_MOD, PM_WKDEP);
	prm_write_mod_reg(0, OMAP3430_DSS_MOD, PM_WKDEP);
	prm_write_mod_reg(0, OMAP3430_NEON_MOD, PM_WKDEP);
	prm_write_mod_reg(0, OMAP3430_CAM_MOD, PM_WKDEP);
	prm_write_mod_reg(0, OMAP3430_PER_MOD, PM_WKDEP);
	if (omap_rev() > OMAP3430_REV_ES1_0) {
		prm_write_mod_reg(0, OMAP3430ES2_SGX_MOD, PM_WKDEP);
		prm_write_mod_reg(0, OMAP3430ES2_USBHOST_MOD, PM_WKDEP);
	} else
		prm_write_mod_reg(0, GFX_MOD, PM_WKDEP);

	if (omap_rev() > OMAP3430_REV_ES1_0) {
		/*
		 * This workaround is needed to prevent SGX and USBHOST from
		 * failing to transition to RET/OFF after a warm reset in OFF
		 * mode. Workaround sets a sleepdep of each of these domains
		 * with MPU, waits for a min 2 sysclk cycles and clears the
		 * sleepdep.
		 */
		cm_write_mod_reg(OMAP3430_CM_SLEEPDEP_PER_EN_MPU,
				 OMAP3430ES2_USBHOST_MOD, OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(OMAP3430_CM_SLEEPDEP_PER_EN_MPU,
				 OMAP3430ES2_SGX_MOD, OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(OMAP3430_CM_SLEEPDEP_PER_EN_MPU,
				 OMAP3430_CAM_MOD, OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(OMAP3430_CM_SLEEPDEP_PER_EN_MPU,
				 OMAP3430_DSS_MOD, OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(OMAP3430_CM_SLEEPDEP_PER_EN_MPU,
				 OMAP3430_PER_MOD, OMAP3430_CM_SLEEPDEP);
		udelay(100);
		cm_write_mod_reg(0, OMAP3430ES2_USBHOST_MOD,
				OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(0, OMAP3430ES2_SGX_MOD,
				OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(0, OMAP3430_CAM_MOD,
				OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(0, OMAP3430_DSS_MOD,
				OMAP3430_CM_SLEEPDEP);
		cm_write_mod_reg(0, OMAP3430_PER_MOD,
				OMAP3430_CM_SLEEPDEP);

		prm_write_mod_reg(0, OMAP3430ES2_SGX_MOD, PM_WKDEP);
		prm_write_mod_reg(0, OMAP3430ES2_USBHOST_MOD, PM_WKDEP);
		prm_write_mod_reg(0, OMAP3430_CAM_MOD, PM_WKDEP);

	} else
		prm_write_mod_reg(0, GFX_MOD, PM_WKDEP);
	/*
	 * Enable interface clock autoidle for all modules.
	 * Note that in the long run this should be done by clockfw
	 */
	cm_write_mod_reg(
		OMAP3430_AUTO_MODEM |
		OMAP3430ES2_AUTO_MMC3 |
		OMAP3430ES2_AUTO_ICR |
		OMAP3430_AUTO_AES2 |
		OMAP3430_AUTO_SHA12 |
		OMAP3430_AUTO_DES2 |
		OMAP3430_AUTO_MMC2 |
		OMAP3430_AUTO_MMC1 |
		OMAP3430_AUTO_MSPRO |
		OMAP3430_AUTO_HDQ |
		OMAP3430_AUTO_MCSPI4 |
		OMAP3430_AUTO_MCSPI3 |
		OMAP3430_AUTO_MCSPI2 |
		OMAP3430_AUTO_MCSPI1 |
		OMAP3430_AUTO_I2C3 |
		OMAP3430_AUTO_I2C2 |
		OMAP3430_AUTO_I2C1 |
		OMAP3430_AUTO_UART2 |
		OMAP3430_AUTO_UART1 |
		OMAP3430_AUTO_GPT11 |
		OMAP3430_AUTO_GPT10 |
		OMAP3430_AUTO_MCBSP5 |
		OMAP3430_AUTO_MCBSP1 |
		OMAP3430ES1_AUTO_FAC | /* This is es1 only */
		OMAP3430_AUTO_MAILBOXES |
		OMAP3430_AUTO_OMAPCTRL |
		OMAP3430ES1_AUTO_FSHOSTUSB |
		OMAP3430_AUTO_HSOTGUSB |
		OMAP3430_AUTO_SAD2D |
		OMAP3430_AUTO_SSI,
		CORE_MOD, CM_AUTOIDLE1);

	cm_write_mod_reg(
		OMAP3430_AUTO_PKA |
		OMAP3430_AUTO_AES1 |
		OMAP3430_AUTO_RNG |
		OMAP3430_AUTO_SHA11 |
		OMAP3430_AUTO_DES1,
		CORE_MOD, CM_AUTOIDLE2);

	if (omap_rev() > OMAP3430_REV_ES1_0) {
		cm_write_mod_reg(
			OMAP3430_AUTO_MAD2D |
			OMAP3430ES2_AUTO_USBTLL,
			CORE_MOD, CM_AUTOIDLE3);
	}

	cm_write_mod_reg(
		OMAP3430_AUTO_WDT2 |
		OMAP3430_AUTO_WDT1 |
		OMAP3430_AUTO_GPIO1 |
		OMAP3430_AUTO_32KSYNC |
		OMAP3430_AUTO_GPT12 |
		OMAP3430_AUTO_GPT1 ,
		WKUP_MOD, CM_AUTOIDLE);

	cm_write_mod_reg(
		OMAP3430_AUTO_DSS,
		OMAP3430_DSS_MOD,
		CM_AUTOIDLE);

	cm_write_mod_reg(
		OMAP3430_AUTO_CAM,
		OMAP3430_CAM_MOD,
		CM_AUTOIDLE);

	cm_write_mod_reg(
		OMAP3430_AUTO_GPIO6 |
		OMAP3430_AUTO_GPIO5 |
		OMAP3430_AUTO_GPIO4 |
		OMAP3430_AUTO_GPIO3 |
		OMAP3430_AUTO_GPIO2 |
		OMAP3430_AUTO_WDT3 |
		OMAP3430_AUTO_UART3 |
		OMAP3430_AUTO_GPT9 |
		OMAP3430_AUTO_GPT8 |
		OMAP3430_AUTO_GPT7 |
		OMAP3430_AUTO_GPT6 |
		OMAP3430_AUTO_GPT5 |
		OMAP3430_AUTO_GPT4 |
		OMAP3430_AUTO_GPT3 |
		OMAP3430_AUTO_GPT2 |
		OMAP3430_AUTO_MCBSP4 |
		OMAP3430_AUTO_MCBSP3 |
		OMAP3430_AUTO_MCBSP2,
		OMAP3430_PER_MOD,
		CM_AUTOIDLE);

	if (omap_rev() > OMAP3430_REV_ES1_0) {
		cm_write_mod_reg(
			OMAP3430ES2_AUTO_USBHOST,
			OMAP3430ES2_USBHOST_MOD,
			CM_AUTOIDLE);
	}

	omap_ctrl_writel(OMAP3430_AUTOIDLE, OMAP2_CONTROL_SYSCONFIG);

	/*
	 * Set all plls to autoidle. This is needed until autoidle is
	 * enabled by clockfw
	 */
	cm_write_mod_reg(1 << OMAP3430_AUTO_IVA2_DPLL_SHIFT,
			 OMAP3430_IVA2_MOD, CM_AUTOIDLE2);
	cm_write_mod_reg(1 << OMAP3430_AUTO_MPU_DPLL_SHIFT,
			 MPU_MOD,
			 CM_AUTOIDLE2);
	cm_write_mod_reg((1 << OMAP3430_AUTO_PERIPH_DPLL_SHIFT),
			 PLL_MOD,
			 CM_AUTOIDLE);
	cm_write_mod_reg(1 << OMAP3430ES2_AUTO_PERIPH2_DPLL_SHIFT,
			 PLL_MOD,
			 CM_AUTOIDLE2);

	/*
	 * Enable control of expternal oscillator through
	 * sys_clkreq. In the long run clock framework should
	 * take care of this.
	 */
	prm_rmw_mod_reg_bits(OMAP_AUTOEXTCLKMODE_MASK,
			     1 << OMAP_AUTOEXTCLKMODE_SHIFT,
			     OMAP3430_GR_MOD,
			     OMAP3_PRM_CLKSRC_CTRL_OFFSET);

	/* setup wakup source */
	prm_write_mod_reg(OMAP3430_EN_GPIO1 |
			  OMAP3430_EN_GPT1 | OMAP3430_EN_GPT12,
			  WKUP_MOD, PM_WKEN);
	/* No need to write EN_IO, that is always enabled */
	prm_write_mod_reg(OMAP3430_EN_GPIO1 | OMAP3430_EN_GPT1 |
			  OMAP3430_EN_GPT12,
			  WKUP_MOD, OMAP3430_PM_MPUGRPSEL);
	/* For some reason IO doesn't generate wakeup event even if
	 * it is selected to mpu wakeup goup */
	prm_write_mod_reg(OMAP3430_IO_EN | OMAP3430_WKUP_EN,
			  OCP_MOD, OMAP3_PRM_IRQENABLE_MPU_OFFSET);

	/* Enable wakeups in PER */
	prm_write_mod_reg(OMAP3430_EN_GPIO2 | OMAP3430_EN_GPIO3 |
			  OMAP3430_EN_GPIO4 | OMAP3430_EN_GPIO5 |
			  OMAP3430_EN_GPIO6 | OMAP3430_EN_UART3 |
			  OMAP3430_EN_MCBSP2 | OMAP3430_EN_MCBSP3 |
			  OMAP3430_EN_MCBSP4,
			  OMAP3430_PER_MOD, PM_WKEN);
	/* and allow them to wake up MPU */
	prm_write_mod_reg(OMAP3430_GRPSEL_GPIO2 | OMAP3430_EN_GPIO3 |
			  OMAP3430_GRPSEL_GPIO4 | OMAP3430_EN_GPIO5 |
			  OMAP3430_GRPSEL_GPIO6 | OMAP3430_EN_UART3,
			  OMAP3430_PER_MOD, OMAP3430_PM_MPUGRPSEL);

	/* Don't attach IVA interrupts */
	prm_write_mod_reg(0, WKUP_MOD, OMAP3430_PM_IVAGRPSEL);
	prm_write_mod_reg(0, CORE_MOD, OMAP3430_PM_IVAGRPSEL1);
	prm_write_mod_reg(0, CORE_MOD, OMAP3430ES2_PM_IVAGRPSEL3);
	prm_write_mod_reg(0, OMAP3430_PER_MOD, OMAP3430_PM_IVAGRPSEL);

	/* Clear any pending 'reset' flags */
	prm_write_mod_reg(0xffffffff, MPU_MOD, RM_RSTST);
	prm_write_mod_reg(0xffffffff, CORE_MOD, RM_RSTST);
	prm_write_mod_reg(0xffffffff, OMAP3430_PER_MOD, RM_RSTST);
	prm_write_mod_reg(0xffffffff, OMAP3430_EMU_MOD, RM_RSTST);
	prm_write_mod_reg(0xffffffff, OMAP3430_NEON_MOD, RM_RSTST);
	prm_write_mod_reg(0xffffffff, OMAP3430_DSS_MOD, RM_RSTST);
	prm_write_mod_reg(0xffffffff, OMAP3430ES2_USBHOST_MOD, RM_RSTST);

	/* Clear any pending PRCM interrupts */
	prm_write_mod_reg(0, OCP_MOD, OMAP3_PRM_IRQSTATUS_MPU_OFFSET);

	omap3_iva_idle();
	omap3_d2d_idle();
}

void omap3_pm_off_mode_enable(int enable)
{
	struct power_state *pwrst;
	u32 state;

	if (enable)
		state = PWRDM_POWER_OFF;
	else
		state = PWRDM_POWER_RET;

#ifdef CONFIG_OMAP_PM_SRF
	resource_lock_opp(VDD1_OPP);
	resource_lock_opp(VDD2_OPP);
	if (resource_refresh())
		printk(KERN_ERR "Error: could not refresh resources\n");
	resource_unlock_opp(VDD1_OPP);
	resource_unlock_opp(VDD2_OPP);
#endif
	list_for_each_entry(pwrst, &pwrst_list, node) {
		pwrst->next_state = state;
		set_pwrdm_state(pwrst->pwrdm, state);
	}
	if (state == PWRDM_POWER_OFF && disable_core_off) {
		core_pwrst->next_state = PWRDM_POWER_RET;
		set_pwrdm_state(core_pwrst->pwrdm, PWRDM_POWER_RET);
	}
}

int omap3_pm_get_suspend_state(struct powerdomain *pwrdm)
{
	struct power_state *pwrst;

	list_for_each_entry(pwrst, &pwrst_list, node) {
		if (pwrst->pwrdm == pwrdm)
			return pwrst->next_state;
	}
	return -EINVAL;
}

int omap3_pm_set_suspend_state(struct powerdomain *pwrdm, int state)
{
	struct power_state *pwrst;

	list_for_each_entry(pwrst, &pwrst_list, node) {
		if (pwrst->pwrdm == pwrdm) {
			pwrst->next_state = state;
			return 0;
		}
	}
	return -EINVAL;
}

static int __init pwrdms_setup(struct powerdomain *pwrdm, void *unused)
{
	struct power_state *pwrst;
	u32 next_state;

	if (!pwrdm->pwrsts)
		return 0;

	/*
	 * Skip IVA2 for handling powerstate conflict between PM and
	 * bridgedriver path.
	 */
	if (strcmp("iva2_pwrdm", pwrdm->name)) {
		pwrst = kmalloc(sizeof(struct power_state), GFP_ATOMIC);
	} else {
	/* IVA2 next powerstate always set to OFF */
		next_state = PWRDM_POWER_OFF;
		goto skip_iva2;
	}

	if (!pwrst)
		return -ENOMEM;
	pwrst->pwrdm = pwrdm;
	if (pwrdm == core_pwrdm)
		core_pwrst = pwrst;

	if (enable_off_mode)
		pwrst->next_state = PWRDM_POWER_OFF;
	else
		pwrst->next_state = PWRDM_POWER_RET;

	list_add(&pwrst->node, &pwrst_list);
	next_state = pwrst->next_state;

skip_iva2:
	if (pwrdm_has_hdwr_sar(pwrdm))
		pwrdm_enable_hdwr_sar(pwrdm);

	return set_pwrdm_state(pwrdm, next_state);
}

/*
 * Enable hw supervised mode for all clockdomains if it's
 * supported. Initiate sleep transition for other clockdomains, if
 * they are not used
 */
static int __init clkdms_setup(struct clockdomain *clkdm, void *unused)
{
	if (clkdm->flags & CLKDM_CAN_ENABLE_AUTO)
		omap2_clkdm_allow_idle(clkdm);
	else if (clkdm->flags & CLKDM_CAN_FORCE_SLEEP &&
		 atomic_read(&clkdm->usecount) == 0)
		omap2_clkdm_sleep(clkdm);
	return 0;
}

void omap_push_sram_idle(void)
{
	_omap_sram_idle = omap_sram_push(omap34xx_cpu_suspend,
					omap34xx_cpu_suspend_sz);
	/* the sdrc counters are always at the end of the omap34xx_cpu_suspend
	 * block */
	_sdrc_counters = (unsigned *)((u8 *)_omap_sram_idle + omap34xx_cpu_suspend_sz - 8);

	if (omap_type() != OMAP2_DEVICE_TYPE_GP)
		_omap_save_secure_sram = omap_sram_push(save_secure_ram_context,
				save_secure_ram_context_sz);
}

static void pm_errata_configure(void)
{
	if (cpu_is_omap34xx()) {
		pm34xx_errata |= PER_WAKEUP_DEP_ERRATAS;
		if (cpu_is_omap3630() && (omap_rev() > OMAP3630_REV_ES1_1))
			pm34xx_errata &= ~PER_WAKEUP_ERRATA_i582;
		if (cpu_is_omap3630())
			pm34xx_errata |= RTA_ERRATA_i608;
	}
}

static int __init omap3_pm_init(void)
{
	struct power_state *pwrst, *tmp;
	unsigned long freq;
	int ret;

	if (!cpu_is_omap34xx())
		return -ENODEV;

	pm_errata_configure();

	printk(KERN_ERR "Power Management for TI OMAP3.\n");

	/*
	 * Some sanity checks here. Panic at fail path, since
	 * once we don't have a valid secure RAM address on Non-GP,
	 * we will hang / crash / reboot by the time we perform OFF
	 * transitions.
	 */
	if (omap_type() != OMAP2_DEVICE_TYPE_GP &&
		!valid_secure_ram_address(__pa(omap3_secure_ram_storage)))
		panic("Got an out of range address for secure RAM context.\n");

	/* XXX prcm_setup_regs needs to be before enabling hw
	 * supervised mode for powerdomains */
	prcm_setup_regs();

	ret = request_irq(INT_34XX_PRCM_MPU_IRQ,
			  (irq_handler_t)prcm_interrupt_handler,
			  IRQF_DISABLED, "prcm", NULL);
	if (ret) {
		printk(KERN_ERR "request_irq failed to register for 0x%x\n",
		       INT_34XX_PRCM_MPU_IRQ);
		goto err1;
	}

	core_pwrdm = pwrdm_lookup("core_pwrdm");
	ret = pwrdm_for_each(pwrdms_setup, NULL);
	if (ret) {
		printk(KERN_ERR "Failed to setup powerdomains\n");
		goto err2;
	}

	/* Core off is currently broken in OMAP3630 (only rev < ES1.2) */
	if (cpu_is_omap3630() && (omap_rev() < OMAP3630_REV_ES1_2)) {
		disable_core_off = 1;
		if (core_pwrst)
			core_pwrst->next_state = PWRDM_POWER_RET;
	}

	(void) clkdm_for_each(clkdms_setup, NULL);

	mpu_pwrdm = pwrdm_lookup("mpu_pwrdm");
	if (mpu_pwrdm == NULL) {
		printk(KERN_ERR "Failed to get mpu_pwrdm\n");
		goto err2;
	}

	neon_pwrdm = pwrdm_lookup("neon_pwrdm");
	per_pwrdm = pwrdm_lookup("per_pwrdm");
	wkup_pwrdm = pwrdm_lookup("wkup_pwrdm");

	omap_push_sram_idle();

	sec_hal_register(&pm_sec_hal_ops);

#ifdef CONFIG_SUSPEND
	suspend_set_ops(&omap_pm_ops);
#endif /* CONFIG_SUSPEND */

	pm_idle = omap3_pm_idle;
	omap3_idle_init();

	pwrdm_add_wkdep(neon_pwrdm, mpu_pwrdm);
	/*
	 * REVISIT: This wkdep is only necessary when GPIO2-6 are enabled for
	 * IO-pad wakeup.  Otherwise it will unnecessarily waste power
	 * waking up PER with every CORE wakeup - see
	 * http://marc.info/?l=linux-omap&m=121852150710062&w=2
	*/
	pwrdm_add_wkdep(per_pwrdm, core_pwrdm);

	/*
	 * RTA is disabled during initialization as per errata i608
	 * it is safer to disable rta by the bootloader, but we would like
	 * to be doubly sure here and prevent any mishaps.
	 */
	if (IS_PM34XX_ERRATA(RTA_ERRATA_i608))
		omap_ctrl_writel(OMAP36XX_RTA_DISABLE,
			OMAP36XX_CONTROL_MEM_RTA_CTRL);

	/* Get lowest and highest L3 OPP for DLL lock WA */
        freq = 0;
        min_opp = opp_find_freq_ceil(OPP_L3, &freq);
        freq = ULONG_MAX;
        max_opp = opp_find_freq_floor(OPP_L3, &freq);

	/*
	 * Enable HW Sup transition for the EMU domain if no emu clocks
	 * are aquired.
	 */
	emu_clkdm = clkdm_lookup("emu_clkdm");
	if (atomic_read(&emu_clkdm->usecount) == 0) {
		prm_write_mod_reg(OMAP_DOMAINWKUP_RST, OMAP3430_EMU_MOD,
								RM_RSTST);
		cm_rmw_mod_reg_bits(OMAP3430_CLKTRCTRL_EMU_MASK,
				(CLKDM_CAN_ENABLE_AUTO | CLKDM_CAN_SWSUP),
				OMAP3430_EMU_MOD, CM_CLKSTCTRL);
	}

	/*
	 * TI recomandation for configuration of CONTROL_PBIAS_LITE
	 * and CONTROL_WKUP_CTRL registers in case of APE_VDDS_SIM shunt
	 * resistor grounded to reduce static leakage current.
	 */
	if (cpu_is_omap3630()) {
		u32 reg;

		reg = omap_ctrl_readl(OMAP343X_CONTROL_PBIAS_LITE);
		reg &= ~OMAP343X_PBIASLITEPWRDNZ1;
		omap_ctrl_writel(reg, OMAP343X_CONTROL_PBIAS_LITE);

		reg = omap_ctrl_readl(OMAP343X_CONTROL_WKUP_CTRL);
		reg &= ~GPIO_IO_PWRDNZ;
		omap_ctrl_writel(reg, OMAP343X_CONTROL_WKUP_CTRL);
	}

	omap3_save_scratchpad_contents();
err1:
	return ret;
err2:
	free_irq(INT_34XX_PRCM_MPU_IRQ, NULL);
	list_for_each_entry_safe(pwrst, tmp, &pwrst_list, node) {
		list_del(&pwrst->node);
		kfree(pwrst);
	}
	return ret;
}

static int __init omap3_pm_early_init(void)
{
	prm_clear_mod_reg_bits(OMAP3430_OFFMODE_POL, OMAP3430_GR_MOD,
				OMAP3_PRM_POLCTRL_OFFSET);
	/* Initializes OMAP3 voltage modules */
	omap_voltage_init();
	return 0;
}

int omap3_check_secure_ram_dirty(void)
{
	if (omap_type() != OMAP2_DEVICE_TYPE_GP)
		return secure_ram_dirty;

	return 0;
}

/**
 * Note: This function is called with interrupts disabled
 */
static void omap3_pm_mark_secure_ram_dirty(void)
{
	secure_ram_dirty = true;
}

void __init pm_alloc_secure_ram(void)
{
	phys_addr_t pa;
	phys_addr_t size = OMAP_SECURE_COPY_AREA_SIZE;
	pa = memblock_alloc_base(size, PAGE_SIZE,
			OMAP3_SECURE_RAM_HIGH_MARK - size);
	BUG_ON(!pa);
	omap3_secure_ram_storage = phys_to_virt(pa);
	omap3_pm_mark_secure_ram_dirty();
}

arch_initcall(omap3_pm_early_init);
late_initcall(omap3_pm_init);

